
`timescale 1ns/1ps
module test;

// Common Tasks
`include "task_common.sv"

parameter DS_NUM = 4;
parameter RATE_5P94  = 2'b10;
parameter RATE_2P97  = 2'b01;
parameter RATE_1P485 = 2'b00;

event scenario_start;

// TX Source Model
logic   [DS_NUM-1:0][9:0]   tx_data_ds;
logic                       tx_frame_start;
logic   [10:0]              tx_ln;
logic                       tx_sav;
logic                       p_level_b;
logic                       p_type2;

// SDI TX
logic                       txdiv2 = 1'b0;
logic   [3:0]               tx_core_rst_gen;
logic                       p_tx_12g;
logic                       tx_core_clk;
logic                       tx_core_rst;
logic   [39:0]              tx_data;
logic   [DS_NUM-1:0][31:0]  tx_vpid;
logic   [DS_NUM-1:0][10:0]  tx_vpid_ln_f1;
logic   [DS_NUM-1:0][10:0]  tx_vpid_ln_f2;

// SDI RX
logic   [39:0]              rx_data;
logic                       rxdiv2 = 1'b0;
logic   [3:0]               rx_core_rst_gen;
logic                       p_rx_12g;
logic                       rx_core_rst;
logic                       rx_core_clk;
logic                       o_align_done;
logic                       o_vd;
logic                       o_hd;
logic   [DS_NUM-1:0][9:0]   o_data_ds;
logic                       o_lock;
logic                       o_rx_type;

// PMA Model
logic   [63:0]              RXD;
logic   [63:0]              TXD;
logic                       RX_CLK;
logic                       TX_CLK;
logic                       RX_RST;
logic                       TX_RST;


// End of automatics

/*AUTOREGINPUT*/
// Beginning of automatic reg inputs (for undeclared instantiated-module inputs)
reg                     i_rst_n;
reg [1:0]               p_sdi_rx_mode;
reg [1:0]               p_sdi_tx_mode;
reg [1:0]               pma_rate;
// End of automatics

//-----------------------------------------------------------------------------
// SDI MODE
//-----------------------------------------------------------------------------
always_comb begin
    case (pma_rate)
        2'b00: begin
            p_sdi_rx_mode <= 2'b00;
            p_sdi_tx_mode <= 2'b00;
        end
        2'b01: begin
            p_sdi_rx_mode <= 2'b00;
            p_sdi_tx_mode <= 2'b00;
        end
        2'b10: begin
            p_sdi_rx_mode <= 2'b01;
            p_sdi_tx_mode <= 2'b01;
        end
        2'b11: begin
            p_sdi_rx_mode <= 2'b10;
            p_sdi_tx_mode <= 2'b10;
        end
        default: begin
            p_sdi_rx_mode <= 2'b10;
            p_sdi_tx_mode <= 2'b10;
        end
    endcase
end

//-----------------------------------------------------------------------------
// TX CORE CLOCK
//-----------------------------------------------------------------------------
always @(posedge TX_CLK)
    txdiv2 <= ~txdiv2;

assign p_tx_12g = p_sdi_tx_mode == 2'b10;
assign tx_core_clk = p_tx_12g ? txdiv2 : TX_CLK;

always @(posedge tx_core_clk or negedge i_rst_n)
    if(!i_rst_n)
        tx_core_rst_gen <= '1;
    else
        tx_core_rst_gen <= {tx_core_rst_gen[2:0], 1'b0};
assign tx_core_rst = tx_core_rst_gen[3];

always @(posedge TX_CLK or negedge i_rst_n)
    if(!i_rst_n)
        TX_RST <= 1'b1;
    else
        TX_RST <= 1'b0;

//-----------------------------------------------------------------------------
// RX CORE CLOCK
//-----------------------------------------------------------------------------
always @(posedge RX_CLK)
    rxdiv2 <= ~rxdiv2;

assign p_rx_12g = p_sdi_rx_mode == 2'b10;
assign rx_core_clk = p_rx_12g ? rxdiv2 : RX_CLK;

always @(posedge rx_core_clk or negedge i_rst_n)
    if(!i_rst_n)
        rx_core_rst_gen <= '1;
    else
        rx_core_rst_gen <= {rx_core_rst_gen[2:0], 1'b0};
assign rx_core_rst = rx_core_rst_gen[3];

always @(posedge RX_CLK or negedge i_rst_n)
    if(!i_rst_n)
        RX_RST <= 1'b1;
    else
        RX_RST <= 1'b0;

//-----------------------------------------------------------------------------
// TX Model
//-----------------------------------------------------------------------------
sdi_tx_model #(
     .DS_NUM                    ( DS_NUM        )
     ) u_sdi_tx_model(
     // Inputs
     .i_clk                     ( tx_core_clk   ),
     .i_rst                     ( tx_core_rst   ),
     .i_level_b                 ( p_level_b     ),
     .i_type2                   ( p_type2       ),
     .i_sdi_mode                ( p_sdi_tx_mode ),
     // Outputs
     .o_data_ds                 ( tx_data_ds    ),
     .o_tx_line                 ( tx_ln         ),
     .o_sav                     ( tx_sav        ),
     .o_frame_start             ( tx_frame_start)
     );   // Templated

sdi_tx_wrapper #(
     // Parameters
     .DS_NUM                    ( DS_NUM        )
     ) u_sdi_tx_wrapper (
     // Inputs
     .i_tx_clk                  ( TX_CLK        ),
     .i_tx_rst                  ( TX_RST        ),
     .i_data_ds                 ( tx_data_ds    ),
     .i_ln                      ( tx_ln         ),
     .i_sdi_mode                ( p_sdi_tx_mode ),
     .ip_rate                   ( pma_rate      ),
     .ip_eq_pattern             ( 1'b0          ),
     .ip_pathological_ptn_en    ( 1'b0          ),
     .ip_vpid                   ( tx_vpid       ),
     .ip_vpid_ln_f1             ( tx_vpid_ln_f1 ),
     .ip_vpid_ln_f2             ( tx_vpid_ln_f2 ),
     // Outputs
     .o_3g_hd_eq_ptn            ( ),
     .o_pathological_ptn        ( ),
     .o_tx_data                 ( tx_data       )
     );

assign TXD[19: 0] = tx_data[19: 0];
assign TXD[31:20] = '0;
assign TXD[51:32] = tx_data[39:20];
assign TXD[63:52] = '0;

pma_model u_pma_model (
     // Inputs
     .i_rst_n                   ( i_rst_n       ),
     .i_rate                    ( pma_rate      ),
     .TXD                       ( TXD[63:0]     ),
     // Outputs
     .TX_CLK                    ( TX_CLK        ),
     .RX_CLK                    ( RX_CLK        ),
     .RXD                       ( RXD           )
     );

assign rx_data  = {RXD[51:32],RXD[19:0]};

sdi_rx_wrapper #(
     // Parameters
     .DS_NUM                    ( DS_NUM            )
     ) u_sdi_rx_wrapper (
     // Inputs
     .i_rx_clk                  ( RX_CLK            ),
     .i_rx_rst                  ( RX_RST            ),
     .i_rx_data                 ( rx_data           ),
     .i_sdi_mode                ( p_sdi_rx_mode     ),
     .i_clear_align_done        ( 1'b0              ),
     .ip_clear_lock             ( 1'b0              ),
     .ip_vpid_cap_on            ( 1'b0              ),
     .ip_crc_error_cnt_clr      ( 1'b0              ),
     .ip_crc_error_cnt_lat      ( 1'b0              ),
     .ip_illegal_code_cnt_clr   ( 1'b0              ),
     .ip_illegal_code_cnt_lat   ( 1'b0              ),
     .ip_frame_cnt_lat          ( 1'b0              ),
     .ip_frame_cnt_clr          ( 1'b0              ),
     // Outputs
     .o_align_done              ( o_align_done      ),
     .o_rx_type                 ( o_rx_type         ),
     .o_lock                    ( o_lock            ),
     .o_vd                      ( o_vd              ),
     .o_hd                      ( o_hd              ),
     .o_data_ds                 ( o_data_ds         ),
     .o_align_timeout           (                   ),
     .op_c_vpid_cap_done        (                   ),
     .op_y_vpid_cap_done        (                   ),
     .op_c_vpid                 (                   ),
     .op_c_vpid_cs_error        (                   ),
     .op_c_vpid_ln              (                   ),
     .op_y_vpid                 (                   ),
     .op_y_vpid_cs_error        (                   ),
     .op_y_vpid_ln              (                   ),
     .o_vpid_chg_irq            (                   ),
     .op_irq_vpid               (                   ),
     .op_crc_error_cnt          (                   ),
     .op_crc_error_det          (                   ),
     .op_illegal_code_cnt       (                   ),
     .op_illegal_code_det       (                   ),
     .op_frame_cnt              (                   ),
     .o_tx_vpid_ln              (                   ),
     .o_tx_vpid                 (                   ),
     .o_tx_xtotal               (                   ),
     .o_tx_xsize                (                   ),
     .o_tx_ytotal               (                   ),
     .o_tx_ysize                (                   )
     );

initial begin
    $timeformat(-9, 3, " ns", 12);
    i_rst_n = 1'b0;
    #200ns;
    i_rst_n = 1'b1;
    ->scenario_start;
end

`include "scenario.sv"
//-----------------------------------------------------------------------------
// Data Checker
//-----------------------------------------------------------------------------
logic           exp_data_en;
logic   [9:0]   exp_data_buf[DS_NUM-1:0][$];
logic   [DS_NUM-1:0][9:0]   source_data;
logic           tx_frame_start_d1;
logic           tx_frame_start_1_pls;

always @(posedge tx_core_clk)
    if(tx_core_rst)
        tx_frame_start_d1 <= 1'b0;
    else
        tx_frame_start_d1 <= tx_frame_start;

assign tx_frame_start_1_pls = ~tx_frame_start & tx_frame_start_d1;  // Neg

always @(posedge tx_core_clk)
    if(tx_core_rst)
        exp_data_en <= 1'b0;
    else if(tx_frame_start_1_pls)
        exp_data_en <= 1'b1;

logic   tx_hactive;
logic   [15:0]  tx_hcnt;
logic   tx_sav_d1;
logic   tx_sav_1_pls;

always @(posedge tx_core_clk)
    if(tx_core_rst)
        tx_sav_d1 <= 1'b0;
    else
        tx_sav_d1 <= tx_sav;

assign tx_sav_1_pls = ~tx_sav & tx_sav_d1; // Neg

always @(posedge tx_core_clk)
    if(tx_core_rst)
        tx_hcnt <= '0;
    else if(tx_sav_1_pls)
        tx_hcnt <= '0;
    else
        tx_hcnt <= &tx_hcnt ? tx_hcnt : tx_hcnt + 1'b1;

logic   [15:0]  p_tx_hactive = 1920;

assign tx_hactive = tx_hcnt < p_tx_hactive;

always @(posedge tx_core_clk)
    if(tx_core_rst)
        source_data <= '0;
    else
        source_data <= tx_data_ds;

always @(posedge tx_core_clk)
    if(exp_data_en & tx_hactive) begin
        if(p_sdi_rx_mode == 2'b10) begin // 12G
            for(int i = 0; i < DS_NUM; i ++ ) begin
                exp_data_buf[i].push_back(source_data[i]);
            end
        end
        else if(p_sdi_rx_mode == 2'b01) begin // 6G
            for(int i = 0; i < DS_NUM; i ++ ) begin
                if(i < 4 ) exp_data_buf[i].push_back(source_data[i]);
                else       exp_data_buf[i].push_back(10'h000);
            end
        end
        else begin // 3G
            for(int i = 0; i < DS_NUM; i ++ ) begin
                if(i < 2 ) exp_data_buf[i].push_back(source_data[i]);
                else       exp_data_buf[i].push_back(10'h000);
            end
        end
    end


logic           rx_hactive;
logic   [15:0]  rx_hcnt;

always @(posedge rx_core_clk)
    if(rx_core_rst)
        rx_hcnt <= '1;
    else if(o_hd)
        rx_hcnt <= '0;
    else
        rx_hcnt <= &rx_hcnt ? rx_hcnt : rx_hcnt + 1'b1;

assign rx_hactive = rx_hcnt < p_tx_hactive;

logic   data_check_on;
always @(posedge rx_core_clk)
    if(rx_core_rst)
        data_check_on <= 1'b0;
    else if(o_vd)
        data_check_on <= 1'b1;


logic   [DS_NUM-1:0][9:0]   exp_data;
logic   [2:0]               exp_sync;
integer data_check_cnt = 0;
always @(posedge rx_core_clk) begin
    if(data_check_on & rx_hactive) begin
        data_check_cnt <= data_check_cnt + 1;
        // Data
        for(int i = 0; i < DS_NUM; i ++) begin
            exp_data[i] = exp_data_buf[i].pop_front();
        end
        for(int i = 0; i < DS_NUM; i ++) begin
            if(exp_data[i] != o_data_ds[i])
                `sim_error($sformatf("DS[%1d] : Data Mismatch, Dut[%3h], Exp[%3h]", i, o_data_ds[i], exp_data[i]));
        end
    end
end

always @(posedge rx_core_clk)
    for(int i = 0; i < DS_NUM; i++) begin
        if(u_sdi_rx_wrapper.y_vpid_valid[i])
            `sim_info($sformatf("VPID DS[%1d] : 0x%08X, LN:%4d", i, u_sdi_rx_wrapper.y_vpid[i], u_sdi_rx_wrapper.ln[i]));
    end


logic [15:0]    rx_line_conter;

always @(posedge rx_core_clk)
    if(rx_core_rst)
        rx_line_conter <= '1;
    else if(o_hd)
        rx_line_conter <= rx_line_conter + 1;

logic   [DS_NUM-1:0][7:0]   crc_error_coutner;

always @(posedge rx_core_clk)
    if(rx_core_rst)
        crc_error_coutner <= '0;
    else if(|u_sdi_rx_wrapper.crc_error)
        crc_error_coutner <= crc_error_coutner + 1'b1;

always @(posedge o_align_done) begin
    `sim_info($sformatf("Align Done!"));
end


always @(posedge o_lock) begin
    `sim_info($sformatf("RX LOCKED!"));
end

initial begin
    @(ev_sim_end);
    `sim_info($sformatf("Total %d line checked", rx_line_conter));
    `sim_info($sformatf("Total %d data checked", data_check_cnt));
    `sim_info($sformatf("CRC Error : %d", crc_error_coutner));
end

endmodule
