/*

Copyright (c) 2018 Alex Forencich

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/

// Language: Verilog 2001

`resetall
`timescale 1ns / 1ps
`default_nettype none

/*
 * AXI4 register (write)
 */
module axi_register_wr #
(
    // Width of data bus in bits
    parameter DATA_WIDTH = 32,
    // Width of address bus in bits
    parameter ADDR_WIDTH = 32,
    // Width of wstrb (width of data bus in words)
    parameter STRB_WIDTH = (DATA_WIDTH/8),
    // Width of ID signal
    parameter ID_WIDTH = 8,
    // Propagate awuser signal
    parameter AWUSER_ENABLE = 0,
    // Width of awuser signal
    parameter AWUSER_WIDTH = 1,
    // Propagate wuser signal
    parameter WUSER_ENABLE = 0,
    // Width of wuser signal
    parameter WUSER_WIDTH = 1,
    // Propagate buser signal
    parameter BUSER_ENABLE = 0,
    // Width of buser signal
    parameter BUSER_WIDTH = 1,
    // AW channel register type
    // 0 to bypass, 1 for simple buffer, 2 for skid buffer
    parameter AW_REG_TYPE = 1,
    // W channel register type
    // 0 to bypass, 1 for simple buffer, 2 for skid buffer
    parameter W_REG_TYPE = 2,
    // B channel register type
    // 0 to bypass, 1 for simple buffer, 2 for skid buffer
    parameter B_REG_TYPE = 1
)
(
    input  wire                     clk,
    input  wire                     rst,

    /*
     * AXI slave interface
     */
    input  wire [ID_WIDTH-1:0]      s_axi_awid,
    input  wire [ADDR_WIDTH-1:0]    s_axi_awaddr,
    input  wire [7:0]               s_axi_awlen,
    input  wire [2:0]               s_axi_awsize,
    input  wire [1:0]               s_axi_awburst,
    input  wire                     s_axi_awlock,
    input  wire [3:0]               s_axi_awcache,
    input  wire [2:0]               s_axi_awprot,
    input  wire [3:0]               s_axi_awqos,
    input  wire [3:0]               s_axi_awregion,
    input  wire [AWUSER_WIDTH-1:0]  s_axi_awuser,
    input  wire                     s_axi_awvalid,
    output wire                     s_axi_awready,
    input  wire [DATA_WIDTH-1:0]    s_axi_wdata,
    input  wire [STRB_WIDTH-1:0]    s_axi_wstrb,
    input  wire                     s_axi_wlast,
    input  wire [WUSER_WIDTH-1:0]   s_axi_wuser,
    input  wire                     s_axi_wvalid,
    output wire                     s_axi_wready,
    output wire [ID_WIDTH-1:0]      s_axi_bid,
    output wire [1:0]               s_axi_bresp,
    output wire [BUSER_WIDTH-1:0]   s_axi_buser,
    output wire                     s_axi_bvalid,
    input  wire                     s_axi_bready,

    /*
     * AXI master interface
     */
    output wire [ID_WIDTH-1:0]      m_axi_awid,
    output wire [ADDR_WIDTH-1:0]    m_axi_awaddr,
    output wire [7:0]               m_axi_awlen,
    output wire [2:0]               m_axi_awsize,
    output wire [1:0]               m_axi_awburst,
    output wire                     m_axi_awlock,
    output wire [3:0]               m_axi_awcache,
    output wire [2:0]               m_axi_awprot,
    output wire [3:0]               m_axi_awqos,
    output wire [3:0]               m_axi_awregion,
    output wire [AWUSER_WIDTH-1:0]  m_axi_awuser,
    output wire                     m_axi_awvalid,
    input  wire                     m_axi_awready,
    output wire [DATA_WIDTH-1:0]    m_axi_wdata,
    output wire [STRB_WIDTH-1:0]    m_axi_wstrb,
    output wire                     m_axi_wlast,
    output wire [WUSER_WIDTH-1:0]   m_axi_wuser,
    output wire                     m_axi_wvalid,
    input  wire                     m_axi_wready,
    input  wire [ID_WIDTH-1:0]      m_axi_bid,
    input  wire [1:0]               m_axi_bresp,
    input  wire [BUSER_WIDTH-1:0]   m_axi_buser,
    input  wire                     m_axi_bvalid,
    output wire                     m_axi_bready
);


`pragma protect begin_protected
`pragma protect version=1
`pragma protect encrypt_agent="ipecrypt"
`pragma protect encrypt_agent_info="http://ipencrypter.com Version: 20.0.8"
`pragma protect author="author-a"
`pragma protect author_info="author-a-details"
`pragma protect data_method="aes256-cbc"

`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIF-SIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_block
Pt3WiJ04zj5bHTr2sAxyNj/2sCSaKrMC231+4XdzPQIPu5irVymbyEsbZoOjpF0U
QnfWyktn4OjIamTFHotl6adcoerVu3o7WlZ2POry3JFKJRBfyRWtdl/K6rRLT15T
YAED8VgdE0GM91CEZP3+A+j0cV3XRvbc2ycNsoqnMJs=
`pragma protect key_keyowner="Aldec"
`pragma protect key_method="rsa"
`pragma protect key_keyname="ALDEC15_001"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_block
bQUduP+QtAbK4d2t7F6iEcuTlgm3GKigGr9kCKW3+SRglW3a3ypTUESbejOCXCLG
H0NVEpG5Yxhx+D9UwwQZOwZDDPbpZ1B7rk3eTEnoK/S7FLV+10rhLARP72P+qqD7
korMhf3NKgav+ENcedzAdPWmJdDJDtOkRlyV2OIBVL17B4x35ZQPeR2B1DHN5qkp
bBG3wOs4eURf+UVB/rrgEuXsJKYvFHeJD933QxtyeqNrUEVva13u3xtQhJMINec2
QdbfgCUQ5XZeb4zKhj5WdiK6+JJ+4pmzRhO/m81QzRRnoRoEZwgKpccJEXRFAKg6
0RM1GyoNwdNXczuRG0tIJA==
`pragma protect key_keyowner="Synopsys"
`pragma protect key_method="rsa"
`pragma protect key_keyname="SNPS-VCS-RSA-2"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_block
gy/tMiXm3UmWtDiGF3b8I/I4UopgFZEJ9aFDovEL7I64qrPa2hu+/Ruiy2Oh9Jsx
2oQFCUaqKT2gCocKP3zJhboxLARaUEKdsW10bPim4Ggouu5X2Y3UkCIexRe+jzJK
JhmAr4ZnjK5+na9Y3ocqfWMW1RL2UTY/VbDKY/gYOsk=
`pragma protect key_keyowner="Cadence Design Systems."
`pragma protect key_method="rsa"
`pragma protect key_keyname="CDS_RSA_KEY_VER_1"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_block
bxNvtak4M3qDwXP12U9dzRNJ4dd7acmip56WzLF5ws+t957WoHZBh1YoEBSrMTB3
N4p2l7b6INmA7z1SdrjFHvT+Eu+h4UAOYlvfeIdhLS6mksQt4rONkEcd1JpH/6an
u/KBIx99xHdkWffnnazl3vNnPucrGAcHvLgFaioxS9+4gNkPNQHrKFkp5ECKIhkN
WRx4qRxwAezXz8Nh9XKOCwIDAK1x86zjMOaLvyA3YCTdPkH2Zpd3i7zrPzUg9QEL
ehmzq7WnBjyFGQt99Nmm1zLLny5G80jkoL1G8+OtZ08nXspIAqshcs3yDmavhFox
sCEjSaIobf6Gwg8Ql0GfYQ==
`pragma protect key_keyowner="Efinix Inc."
`pragma protect key_keyname="EFX_K01"
`pragma protect key_method="rsa"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_block
KxZbedKPFwxtaaqey51GiULp5SmMiMN+01jOARik8zd2VJWZo+NhrdlHThtXyUFc
jopVA0JFw7PC1r8Q7MF1ppOb8S/qMA8R2DHL/zzMJ7NX1yIxfPV6yt+Ft0GOMnq3
ojTkVrx7jajPhcjcGi/9EJUITel3fRp8yPaL5ngk/1gjBzOhWhM8ejKi7AHPIf2O
THJBMnVMXdPeZIQHgK42Zs9wKr6zS67FJg616B7j9BDe51b4EZxCE92CkgAcYc96
xQlhNZsMv/TS3xMDQRReKWEkV8CCZgS7CoTAibXTjCI0fPYD2QSLdDm6mR+IFFHV
Vtd+FwpuhpRm/ZoVfVqTKg==
`pragma protect encoding=(enctype="base64", line_length=64, bytes=19296)
`pragma protect data_block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`pragma protect end_protected
endmodule

`resetall
