/*

Copyright (c) 2018 Alex Forencich

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/

// Language: Verilog 2001

`resetall
`timescale 1ns / 1ps
`default_nettype none

/*
 * AXI4 register (read)
 */
module axi_register_rd #
(
    // Width of data bus in bits
    parameter DATA_WIDTH = 32,
    // Width of address bus in bits
    parameter ADDR_WIDTH = 32,
    // Width of wstrb (width of data bus in words)
    parameter STRB_WIDTH = (DATA_WIDTH/8),
    // Width of ID signal
    parameter ID_WIDTH = 8,
    // Propagate aruser signal
    parameter ARUSER_ENABLE = 0,
    // Width of aruser signal
    parameter ARUSER_WIDTH = 1,
    // Propagate ruser signal
    parameter RUSER_ENABLE = 0,
    // Width of ruser signal
    parameter RUSER_WIDTH = 1,
    // AR channel register type
    // 0 to bypass, 1 for simple buffer, 2 for skid buffer
    parameter AR_REG_TYPE = 1,
    // R channel register type
    // 0 to bypass, 1 for simple buffer, 2 for skid buffer
    parameter R_REG_TYPE = 2
)
(
    input  wire                     clk,
    input  wire                     rst,

    /*
     * AXI slave interface
     */
    input  wire [ID_WIDTH-1:0]      s_axi_arid,
    input  wire [ADDR_WIDTH-1:0]    s_axi_araddr,
    input  wire [7:0]               s_axi_arlen,
    input  wire [2:0]               s_axi_arsize,
    input  wire [1:0]               s_axi_arburst,
    input  wire                     s_axi_arlock,
    input  wire [3:0]               s_axi_arcache,
    input  wire [2:0]               s_axi_arprot,
    input  wire [3:0]               s_axi_arqos,
    input  wire [3:0]               s_axi_arregion,
    input  wire [ARUSER_WIDTH-1:0]  s_axi_aruser,
    input  wire                     s_axi_arvalid,
    output wire                     s_axi_arready,
    output wire [ID_WIDTH-1:0]      s_axi_rid,
    output wire [DATA_WIDTH-1:0]    s_axi_rdata,
    output wire [1:0]               s_axi_rresp,
    output wire                     s_axi_rlast,
    output wire [RUSER_WIDTH-1:0]   s_axi_ruser,
    output wire                     s_axi_rvalid,
    input  wire                     s_axi_rready,

    /*
     * AXI master interface
     */
    output wire [ID_WIDTH-1:0]      m_axi_arid,
    output wire [ADDR_WIDTH-1:0]    m_axi_araddr,
    output wire [7:0]               m_axi_arlen,
    output wire [2:0]               m_axi_arsize,
    output wire [1:0]               m_axi_arburst,
    output wire                     m_axi_arlock,
    output wire [3:0]               m_axi_arcache,
    output wire [2:0]               m_axi_arprot,
    output wire [3:0]               m_axi_arqos,
    output wire [3:0]               m_axi_arregion,
    output wire [ARUSER_WIDTH-1:0]  m_axi_aruser,
    output wire                     m_axi_arvalid,
    input  wire                     m_axi_arready,
    input  wire [ID_WIDTH-1:0]      m_axi_rid,
    input  wire [DATA_WIDTH-1:0]    m_axi_rdata,
    input  wire [1:0]               m_axi_rresp,
    input  wire                     m_axi_rlast,
    input  wire [RUSER_WIDTH-1:0]   m_axi_ruser,
    input  wire                     m_axi_rvalid,
    output wire                     m_axi_rready
);


`pragma protect begin_protected
`pragma protect version=1
`pragma protect encrypt_agent="ipecrypt"
`pragma protect encrypt_agent_info="http://ipencrypter.com Version: 20.0.8"
`pragma protect author="author-a"
`pragma protect author_info="author-a-details"
`pragma protect data_method="aes256-cbc"

`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIF-SIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_block
AcsLI+WVKtX7Rbvvpv627MmUoLT0iAwbcx66ZQMIDvErH6Fwg1g5fU4rwmHraj9d
5xCJCLm/mh8PSnmFvLvbIxjn8sW7MxrqCDCPWsbJIwSL3WYItbswRfoMmGJISfSu
4PgoyGmnvlKz5vhf3RgUMB3F72YFnKVVdrytIUDnqO4=
`pragma protect key_keyowner="Aldec"
`pragma protect key_method="rsa"
`pragma protect key_keyname="ALDEC15_001"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_block
r8O0aPILM/tthB8eFufwdhzr2J8iP6pG3AubML16Y4c6sDQvKn8iqs24HJUcpRCE
JQ+BMbM5MOCpG6ue8lIweqEsQOlnQX02pbfI2I4RFudcLQctwcJzgz9BmigRDP9R
mTKdDVaGLNy46qsKSIyZwvgBwhUI/w6jIovWMPUtIDXzizcqnmMRSjpTe4z2gU8u
bFWc6K+VCQezJmkfeAB5X/1IdJZSyFZ/gynKZMHkA5B8CiT8O0ZojpmvPEhQ/vlI
eQS4oXY7ULD0BA4qNb97n1xbFAUTCzJ5Xe/DcKnngDoDdKQD2VOFjmRjQwVALA3f
9NzCJMuZjcvFyxKarttX1Q==
`pragma protect key_keyowner="Synopsys"
`pragma protect key_method="rsa"
`pragma protect key_keyname="SNPS-VCS-RSA-2"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_block
D23Yk7MMMct6VVLcGuF3MJcueT96torxO8qYldj886OeK+X1QF4s/Q9N3rEi+dwB
KstzNfAQUVPXUJxo56dAel01qjBvWr/P6aIYNeBBu71Y6iWBlNNLxYltgsF8NLsh
OMoU4rJfZz9tL5txr4KyjzNRXLlf/BEjGThDa9BdG78=
`pragma protect key_keyowner="Cadence Design Systems."
`pragma protect key_method="rsa"
`pragma protect key_keyname="CDS_RSA_KEY_VER_1"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_block
Gf8k0wm52nl9Bd6NuJ44ewGq2jrh95lhVwjGldlKUwIUcsH1xuGfexRk2zXIgFDj
yq+ycweE7jW5Si8cnlz+n163UgsSrUuDfc5rUYpjgN4bCRyJK2eJuLeJ7Z7gXohy
WwpuhzvDKajFEiBFYO/6bSqPguU0c+s8SF3mPoazAh8/keeKCodViDRUXVqVeDBF
Ug+D4AZAbuUjLcAN+y5hd4KmpFciI7x5XHiNwibCev1T9p9yC2Hw2ABgPHuEpisR
s8s7O8EQB/FzklajeOJManhV/yW75b8QvLoIg+9gcjq4hVMOMDs1d69DHa+9hyUd
K5/7/3ieIXRlDwk+U1Acjw==
`pragma protect key_keyowner="Efinix Inc."
`pragma protect key_keyname="EFX_K01"
`pragma protect key_method="rsa"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_block
HHfqmDtWO0ecHA4cI6kBpNGhbH2auCN7IAHTuwTzRdZAidkhm8SG5Q9OMu2vssQJ
3ADWsKQF/v2TQprIyBPOuejMdGkwLb52vjng7/t9aBs3X2nlkMWfSYdk8PdQ/uwc
1U6UraYL1BB4NFJ9i1P9ecgba5CRAHh+eACVg4Xoww6MYTnUYhS0JVdEbzxOMsIT
qXOMXsvCTmmKoeDuVD2tfaLsU0UQSfg3vzg1SiQUaVVqKgCUjBFILBcGKnInB2xO
N3b2JPrCqhfFNuFNKoekLGhE99CAl5APZHxQ4WNRK/GJlYjB0pLUlN41PFZFHGs5
PREgTF4Hj89yqNsTd5Qgnw==
`pragma protect encoding=(enctype="base64", line_length=64, bytes=14768)
`pragma protect data_block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`pragma protect end_protected
endmodule

`resetall
