/*
 * Device debug method kset and ktype implementation
 *
 */
#include "dma.h"
#include "common.h"
#include "misc.h"

static struct kobject *g_sysfs_kobject = NULL;
void *bypass_buf_vir;
dma_addr_t bypass_buf_dma;
void misc_bypass_test(struct efx_pci_dev *epdev, int dir);

static ssize_t reg_show(struct efx_pci_dev *epdev, struct efx_attr *attr,
								char *buf)
{
	int rc = 0;
	return rc;
}

static ssize_t reg_store(struct efx_pci_dev *epdev, struct efx_attr *attr,
								 const char *buf, size_t count)
{
	char *p, *this_opt;
	unsigned int reg_oft, reg_val;
	void *reg = NULL;

	while ((this_opt = strsep((char **)&buf, ",")) != NULL)
	{
		if (!(p = strsep(&this_opt, "=")) || !*p)
			return count;
		reg_oft = simple_strtoul(p, NULL, 16);
		if (!(p = strsep(&this_opt, "")) || !*p)
			continue;
		if ('?' == *p)
		{
			reg = epdev->bar[0] + reg_oft;
			reg_val = ioread32(reg);
			pr_info("SYS rd reg[0x%x] val[0x%x]\n", reg_oft, reg_val);
		}
		else
		{
			reg_val = simple_strtoul(p, NULL, 16);

			reg = epdev->bar[0] + reg_oft;
			pr_info("SYS wr reg[0x%x] val[0x%x]\n", reg_oft, reg_val);
			iowrite32(reg_val, reg);
		}
	}
	return count;
}

static struct efx_attr reg_attribute =
	 __ATTR(reg, 0664, reg_show, reg_store);

static ssize_t test_show(struct efx_pci_dev *epdev, struct efx_attr *attr,
								 char *buf)
{
	ssize_t rc = 0;
#if 0
	uint64_t msecs;
	double bw = 0.0;
	uint64_t bw_1 = 0;
	if (trans_jiffs && trans_size) {
		msecs = jiffies64_to_msecs(trans_jiffs);
		pr_info("[LINC]>>> trans_size=%lld, trans_jiffs=%lld, msecs=%lld\n", trans_size, trans_jiffs, msecs);
		bw = (double)(trans_size / 1024 / 1024) / msecs;
		bw_1 = ((trans_size*10) / 1024 / 1024) / msecs;
		//pr_info("Current Bandwidth : %.2fGBPS\n", bw);
		rc += sprintf(buf + rc, "Current Bandwidth : %03lldGBPS\n", bw_1);
		rc += sprintf(buf + rc, "Current Bandwidth : %.2fGBPS\n", bw);
		
	} else {
		rc += sprintf(buf + rc, "Current Bandwidth is unknown!\n");
	}

	uint64_t dbg_first_desc_addr;
	void *access;
	uint32_t *desc;
	
	if (dbg_first_desc_lo_addr || dbg_first_desc_hi_addr) {
		dbg_first_desc_addr = dbg_first_desc_hi_addr;
		dbg_first_desc_addr = dbg_first_desc_addr << 32;
		dbg_first_desc_addr |= dbg_first_desc_lo_addr;
		rc += sprintf(buf + rc, "First descriptor dma address=0x%lx(%p), adjacent=%d\n", dbg_first_desc_addr, desc_addr, dbg_first_desc_adj);
		desc = (uint32_t *)desc_addr;
		rc += sprintf(buf + rc, "First descriptor control=0x%x\n", *desc);
		rc += sprintf(buf + rc, "First descriptor len=0x%x\n", *(desc + 1));
		rc += sprintf(buf + rc, "First descriptor src addr=0x%x%x\n", *(desc + 3), *(desc + 2));
		rc += sprintf(buf + rc, "First descriptor dest addr=0x%x%x\n", *(desc + 5), *(desc + 4));
		rc += sprintf(buf + rc, "First descriptor next addr=0x%x%x\n", *(desc + 7), *(desc + 6));
	}
	if (dbg_poll_mode_addr_virt)
		rc += sprintf(buf + rc, "Compl_descriptor_count[23:0]=0x%x\n", *dbg_poll_mode_addr_virt);
#endif
	return rc;
}

/******************************* BYPASS Demo *********************/
#define DESC_BYPASS_CTRL_R 0x40004
#define H2D_DESC_SRC_ADDR_L_R 0x40008
#define H2D_DESC_SRC_ADDR_H_R 0x4000c
#define D2H_DESC_SRC_ADDR_L_R 0x40010
#define D2H_DESC_SRC_ADDR_H_R 0x40014
#define BYPASS_DIR_H2D 0
#define BYPASS_DIR_D2H 1

void misc_bypass_test(struct efx_pci_dev *epdev, int dir)
{
	int timeout = 1000000;
	unsigned int reg_val;
	void *reg = NULL;
	struct edma_poll_wb *wb_data;
	struct edma_engine *engine;

	reg_val = bypass_buf_dma & 0xFFFFFFFF;
	pr_info("BYPASS set address Low32bit=0x%08x\n", reg_val);
	reg = epdev->bar[0] + H2D_DESC_SRC_ADDR_L_R;
	iowrite32(reg_val, reg);
	reg = epdev->bar[0] + D2H_DESC_SRC_ADDR_L_R;
	iowrite32(reg_val, reg);

	reg_val = (bypass_buf_dma >> 32) & 0xFFFFFFFF;
	pr_info("BYPASS set address High32bit=0x%08x\n", reg_val);
	reg = epdev->bar[0] + H2D_DESC_SRC_ADDR_H_R;
	iowrite32(reg_val, reg);
	reg = epdev->bar[0] + D2H_DESC_SRC_ADDR_H_R;
	iowrite32(reg_val, reg);

	reg_val = (uint32_t)EDMA_CTRL_RUN_STOP;
	reg_val |= (uint32_t)EDMA_CTRL_POLL_MODE_WB;
	pr_info("BYPASS set engine config val=0x%x\n", reg_val);
	// enable poll mode and enable run bit
	if (dir == BYPASS_DIR_H2D)
	{
		engine = &epdev->engine_htc[0];
		wb_data = (struct edma_poll_wb *)engine->poll_mode_addr_virt;
		wb_data->completed_desc_count = 0;

		reg = epdev->bar[0] + 0x04;
		iowrite32(reg_val, reg); // enable H2D running bit & poll mode
		reg_val = 0;
		reg = epdev->bar[0] + DESC_BYPASS_CTRL_R;
		iowrite32(reg_val, reg);

		reg_val = 1;
		reg = epdev->bar[0] + DESC_BYPASS_CTRL_R;
		iowrite32(reg_val, reg);
	}
	else
	{ // BYPASS_DIR_D2H
		engine = &epdev->engine_cth[0];
		wb_data = (struct edma_poll_wb *)engine->poll_mode_addr_virt;
		wb_data->completed_desc_count = 0;

		reg = epdev->bar[0] + 0x1004;
		iowrite32(reg_val, reg); // enable H2D running bit & poll mode

		reg_val = 0;
		reg = epdev->bar[0] + DESC_BYPASS_CTRL_R;
		iowrite32(reg_val, reg);

		reg_val = 2;
		reg = epdev->bar[0] + DESC_BYPASS_CTRL_R;
		iowrite32(reg_val, reg);
	}

	// wait for polling done
	do
	{
		timeout--;
	} while (!wb_data->completed_desc_count && timeout);
	pr_info("BYPASS %s test done! wb count=%d\n", dir ? "D2H" : "H2D", wb_data->completed_desc_count);
	// reg_val = ioread32(reg);
	return;
}

/******************************* BYPASS Demo *********************/

static ssize_t test_store(struct efx_pci_dev *epdev, struct efx_attr *attr,
								  const char *buf, size_t count)
{
	if ('1' == *buf)
	{
		pr_info("Begining to wake up transfer waiting queue.\n");
	}
	else if (!strncmp(buf, "htc", 3))
	{
		misc_bypass_test(epdev, BYPASS_DIR_H2D);
	}
	else if (!strncmp(buf, "cth", 3))
	{
		misc_bypass_test(epdev, BYPASS_DIR_D2H);
	}
	return count;
}

/* Sysfs attributes cannot be world-writable. */
static struct efx_attr test_attribute =
	 __ATTR(test, 0664, test_show, test_store);

static ssize_t bypass_show(struct efx_pci_dev *epdev, struct efx_attr *attr,
									char *buf)
{
	ssize_t rc = 0;
	uint8_t *print_buf;
	int i;
	rc += sprintf(buf + rc, "32KB DMA address: 0x%llx\n", bypass_buf_dma);
	if (bypass_buf_vir)
	{
		print_buf = (uint8_t *)bypass_buf_vir;
		// rc += sprintf(buf + rc, "32KB DMA data:{0x%02x, 0x%02x, 0x%02x, 0x%02x}\n",
		//		print_buf[0], print_buf[1], print_buf[2], print_buf[3]);
		rc += sprintf(buf + rc, "32KB DMA data:\n");
		for (i = 0; i < 128; i++)
		{
			rc += sprintf(buf + rc, "%02x ", print_buf[i]);
			if ((i + 1) % 8 == 0)
				rc += sprintf(buf + rc, "\n ");
		}
		rc += sprintf(buf + rc, "]\n");
	}
	return rc;
}

static ssize_t bypass_store(struct efx_pci_dev *epdev, struct efx_attr *attr,
									 const char *buf, size_t count)
{
	uint32_t test_data;
	if (bypass_buf_vir && bypass_buf_dma)
	{
		int i;
		uint8_t *cpy = (uint8_t *)bypass_buf_vir;
		test_data = simple_strtoul(buf, NULL, 16);
		// pr_info("[BYPASS] test_data=0x%x\n", test_data);
		for (i = 0; i < (32 * 1024); i++)
			cpy[i] = (test_data + i) & 0xFF;
	}
	return count;
}

/* Sysfs attributes cannot be world-writable. */
static struct efx_attr bypass_attribute =
	 __ATTR(bypass, 0664, bypass_show, bypass_store);

static ssize_t msix_show(struct efx_pci_dev *epdev, struct efx_attr *attr,
								 char *buf)
{
	int i, rc = 0;
	unsigned int reg_oft;
	void *reg = NULL;

	reg = epdev->bar[0] + 0x8000;

	for (i = 0; i < 32; i++)
	{
		reg_oft = i * 0x10;
		rc += sprintf(buf + rc, "MSI-X vector[%d] addr=0x%08x %08x, data=0x%08x, control=0x%08x\n", i,
						  ioread32(reg + reg_oft + 0x04), ioread32(reg + reg_oft),
						  ioread32(reg + reg_oft + 0x08), ioread32(reg + reg_oft + 0x0C));
	}
	if (i == 0)
		rc += sprintf(buf + rc, "There is no MSI-X vectors!\n");
	return rc;
}

static ssize_t msix_store(struct efx_pci_dev *epdev, struct efx_attr *attr,
								  const char *buf, size_t count)
{
	unsigned int reg_oft;
	void *reg = NULL;
	reg = epdev->bar[0] + 0x8000;
	if ('0' == *buf)
	{
		int i;
		for (i = 0; i < 32; i++)
		{
			reg_oft = i * 0x10;
			iowrite32(0, reg + reg_oft);
			iowrite32(0, reg + reg_oft + 0x04);
			iowrite32(0, reg + reg_oft + 0x08);
			iowrite32(0, reg + reg_oft + 0x0C);
		}
	}
	else
	{
		uint32_t msi_addr = 0;
		// H2D msi-x configuration
		pci_read_config_dword(epdev->pdev, 0x94, &msi_addr);
		iowrite32(msi_addr, reg + reg_oft);
		pci_read_config_dword(epdev->pdev, 0x98, &msi_addr);
		iowrite32(msi_addr, reg + reg_oft + 0x04);
		iowrite32(0, reg + reg_oft + 0x08);
		iowrite32(0, reg + reg_oft + 0x0C);
		// D2H msi-x configuration
		pci_read_config_dword(epdev->pdev, 0x94, &msi_addr);
		iowrite32(msi_addr, reg + reg_oft + 0x10);
		pci_read_config_dword(epdev->pdev, 0x98, &msi_addr);
		iowrite32(msi_addr, reg + reg_oft + 0x14);
		iowrite32(1, reg + reg_oft + 0x18);
		iowrite32(0, reg + reg_oft + 0x1C);
		pr_info("Configuration for MSI done!\n");
	}
	return count;
}

/* Sysfs attributes cannot be world-writable. */
static struct efx_attr msix_attribute =
	 __ATTR(msix, 0664, msix_show, msix_store);

/*
 * The default show function that must be passed to sysfs.  This will be
 * called by sysfs for whenever a show function is called by the user on a
 * sysfs file associated with the kobjects we have registered.  We need to
 * transpose back from a "default" kobject to our custom struct foo_obj and
 * then call the show function for that specific object.
 */
static ssize_t misc_attr_show(struct kobject *kobj,
										struct attribute *attr,
										char *buf)
{
	struct efx_attr *eattr;
	struct efx_pci_dev *epdev;

	eattr = TO_EFX_ATTR(attr);
	epdev = TO_EFX_PDEV(kobj);

	if (!eattr->show)
		return -EIO;

	return eattr->show(epdev, eattr, buf);
}

static ssize_t misc_attr_store(struct kobject *kobj,
										 struct attribute *attr,
										 const char *buf, size_t len)
{
	struct efx_attr *eattr;
	struct efx_pci_dev *epdev;

	eattr = TO_EFX_ATTR(attr);
	epdev = TO_EFX_PDEV(kobj);

	if (!eattr->store)
		return -EIO;

	return eattr->store(epdev, eattr, buf, len);
}

static const struct sysfs_ops misc_sysfs_ops = {
	 .show = misc_attr_show,
	 .store = misc_attr_store,
};

static void misc_release(struct kobject *kobj)
{
	return;
}

static struct attribute *misc_default_attrs[] = {
	 &reg_attribute.attr,
	 &test_attribute.attr,
	 &bypass_attribute.attr,
	 &msix_attribute.attr,
	 NULL, /* need to NULL terminate the list of attributes */
};
#if KERNEL_VERSION(5, 6, 0) <= LINUX_VERSION_CODE
ATTRIBUTE_GROUPS(misc_default);
#endif

static struct kobj_type misc_ktype = {
	 .sysfs_ops = &misc_sysfs_ops,
	 .release = misc_release,
#if KERNEL_VERSION(5, 6, 0) <= LINUX_VERSION_CODE
	 .default_groups = misc_default_groups,
#else
	 .default_attrs = misc_default_attrs,
#endif
};

int add_kobject(struct efx_pci_dev *epdev, const char *name)
{
	int ret;

	if (!epdev || !g_sysfs_kobject)
		return -ENODEV;
	ret = kobject_init_and_add(&epdev->kobj, &misc_ktype, g_sysfs_kobject, "%s-%d", name, epdev->idx);
	if (ret)
	{
		kobject_put(&epdev->kobj);
		return -EINVAL;
	}

	kobject_uevent(&epdev->kobj, KOBJ_ADD);

	if (!bypass_buf_vir)
		bypass_buf_vir = dma_alloc_coherent(&epdev->pdev->dev, 32 * 1024, &bypass_buf_dma, GFP_KERNEL);
	if (!bypass_buf_vir)
		pr_err("[ERROR] Can not allocate DMA 32KB buffer.\n");
	return 0;
}

void remove_kobject(struct efx_pci_dev *epdev)
{
	if (bypass_buf_vir)
		dma_free_coherent(&epdev->pdev->dev, 32 * 1024, bypass_buf_vir, bypass_buf_dma);
	bypass_buf_vir = NULL;
	kobject_del(&epdev->kobj);
}

void register_sysfs(void)
{
	g_sysfs_kobject = kobject_create_and_add("efinix", NULL);
	if (!g_sysfs_kobject)
	{
		pr_err("[ERROR] Failed to create sysfs kobject\n");
	}
}

void remove_sysfs(void)
{
	kobject_del(g_sysfs_kobject);
}
