----------------------------------------------------------------------------------------------------
-- Copyright (c) ReFLEX CES 1998-2012
--
-- Use of this source code through a simulator and/or a compiler tool
-- is illegal if not authorised through ReFLEX CES License agreement.
----------------------------------------------------------------------------------------------------
-- Project     : 
----------------------------------------------------------------------------------------------------
-- Top level   : top.vhd
-- File        : pdu_chk.vhd
-- Author      : flavenant@reflexces.com
-- Company     : ReFLEX CES
--               2, rue du gevaudan
--               91047 LISSES
--               FRANCE
--               http://www.reflexces.com
-- Plateforme  : Windows XP
-- Simulator   : Mentor Graphics ModelSim
-- Synthesis   : Quartus II
-- Target      : Stratix IV
-- Dependency  :
----------------------------------------------------------------------------------------------------
-- Description :
-- PDU checker.
--  
----------------------------------------------------------------------------------------------------
-- Version      Date            Author              Description
-- 0.1          2012/01/16      FLA                 Creation
----------------------------------------------------------------------------------------------------
library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;
use     ieee.std_logic_unsigned.all;

entity pdu_chk is
    generic (
      NB_16B_LANES   : integer   :=  1; -- Number of 16bits lanes.
      COUNTER_WIDTH  : integer   := 32; -- Number of bits for the frame counter
      streaming_mode : std_logic := '0' -- Streaming Mode if '1', Framing if '0'
    );
    port (
        -- Reset and clocks
          rst_n                 : in    std_logic
        ; clk                   : in    std_logic
        
        -- PDU
        ; pdu_d                 : in    std_logic_vector(16*NB_16B_LANES-1 downto 0)
        ; pdu_vld               : in    std_logic
        ; pdu_bena              : in    std_logic_vector(2*NB_16B_LANES-1 downto 0)
        ; pdu_sop               : in    std_logic
        ; pdu_eop               : in    std_logic
        ; pdu_err               : in    std_logic
        
        -- Control / Status
        ; frame_cnt             : out   std_logic_vector(COUNTER_WIDTH-1 downto 0)      -- number of frames received
        ; error_cnt             : out   std_logic_vector(COUNTER_WIDTH-1 downto 0)      -- number of errors received
        ; error_link_cnt        : out   std_logic_vector(COUNTER_WIDTH-1 downto 0)      -- number of errors received
        ; reset_cnt             : in    std_logic
    );
end entity pdu_chk;

architecture rtl of pdu_chk is

	----------------------------------------------------------------
	-- Signal declarations
	----------------------------------------------------------------
    --synthesis translate_off
    signal s_stop_sim       : std_logic;
    --synthesis translate_on
    
    signal s_sop_cnt        : std_logic_vector(31 downto 0);
    signal s_eop_cnt        : std_logic_vector(31 downto 0);
    signal s_ok_cnt         : std_logic_vector(31 downto 0);
    signal s_data_byte      : std_logic_vector( 7 downto 0);
    signal s_in_frame       : std_logic;
    signal s_frame_cnt      : std_logic_vector(COUNTER_WIDTH-1 downto 0);
    signal s_error          : std_logic;
    signal s_error_cnt      : std_logic_vector(COUNTER_WIDTH-1 downto 0);
    signal s_error_link     : std_logic;
    signal s_error_link_cnt : std_logic_vector(COUNTER_WIDTH-1 downto 0);
    
    signal s_pdu_d          : std_logic_vector(16*NB_16B_LANES-1 downto 0);
    signal s_pdu_vld        : std_logic;
    signal s_pdu_bena       : std_logic_vector( 2*NB_16B_LANES-1 downto 0);
    signal s_pdu_sop        : std_logic;
    signal s_pdu_eop        : std_logic;
    signal s_pdu_err        : std_logic;
    
begin

    --##############################################################
    -- Main process
    --##############################################################  
    p_main : process (clk, rst_n)
        variable v_bena         : std_logic_vector(2*NB_16B_LANES-1 downto 0);
        variable v_data_byte    : std_logic_vector(s_data_byte'range);
        variable v_din_d        : std_logic_vector(s_pdu_d'range);
        variable v_in_frame     : std_logic;
    begin
    if rst_n='0' then 
        --synthesis translate_off
        s_stop_sim       <= '0';
        --synthesis translate_on
        s_sop_cnt        <= (others=>'0');
        s_eop_cnt        <= (others=>'0');
        s_ok_cnt         <= (others=>'0');
        s_data_byte      <= (others=>'0');
        s_in_frame       <= '0';
        s_frame_cnt      <= (others=>'0');
        s_error          <= '0';
        s_error_link     <= '0';
        s_error_cnt      <= (others=>'0');
        s_error_link_cnt <= (others=>'0');
        
    elsif rising_edge(clk) then
    
        -- Default values
        s_error      <= '0';
        s_error_link <= '0';
    
        ----------------------------------------------------------------------------------------------------
        -- Check packet data
        ----------------------------------------------------------------------------------------------------
        -- Check byte enable validates almost one byte
        if s_pdu_vld='1' and s_pdu_eop='1' then
            if s_pdu_bena=0 then
                -- synthesis translate_off
                report "[pdu_chk]: bena=0 is not allowed." severity warning;
                s_stop_sim <= '1';
                -- synthesis translate_on
                s_error <= '1';
            end if;
        end if;
            
        -- Format byte enable : always all checked in Streaming Mode
        v_bena        := s_pdu_bena;
        if s_pdu_eop='0' or streaming_mode='1' then v_bena := (others=>'1'); end if;
        
        -- Generate data bytes
        v_data_byte := s_data_byte;
        v_din_d     := (others=>'0');
        for i in 0 to 2*NB_16B_LANES-1 loop
            if v_bena(i)='1' then v_din_d(i*8+8-1 downto i*8) := v_data_byte + i;
            else                  v_din_d(i*8+8-1 downto i*8) := s_pdu_d(i*8+8-1 downto i*8);
            end if;
        end loop;
        if s_pdu_vld='1' then
            for i in 0 to 2*NB_16B_LANES-1 loop
                if v_bena(i)='1' then v_data_byte := v_data_byte + 1; end if;
            end loop;
            s_data_byte <= v_data_byte;
        end if;
        
        -- Compare values (only if pdu_err = '0') 
        if s_pdu_vld='1' and s_pdu_err = '0' then
            if v_din_d/=s_pdu_d then
                -- synthesis translate_off
                report "[pdu_chk]: invalid data received." severity warning;
                s_stop_sim <= '1';
                -- synthesis translate_on
                s_error     <= '1';
                s_data_byte <= s_pdu_d(7 downto 0) + 2*NB_16B_LANES;
            else
                s_ok_cnt    <= s_ok_cnt + 1;
            end if;
        end if;
        
        -- Link Errors 
        if s_pdu_vld='1' and s_pdu_err = '1' then
          s_error_link <= '1';
        end if;
        
        ----------------------------------------------------------------------------------------------------
        -- Check packet flag
        ----------------------------------------------------------------------------------------------------
        -- Count flags received
        if s_pdu_vld='1' then
            if s_pdu_sop='1' then s_sop_cnt <= s_sop_cnt + 1; end if;
            if s_pdu_eop='1' then s_eop_cnt <= s_eop_cnt + 1; end if;
        end if;
        
        -- Check SOP is received out of frame
        v_in_frame := s_in_frame;
        if s_pdu_vld='1' and s_pdu_sop='1' then
            if v_in_frame='1' then 
                -- synthesis translate_off
                report "[pdu_chk]: SOP received in frame." severity warning;
                s_stop_sim <= '1';
                -- synthesis translate_on
                s_error <= '1';
            else
                v_in_frame := '1';
            end if;
        end if;
        
        -- Check data are received in frame : only for Framing Mode
        if s_pdu_vld='1' and s_pdu_sop='0' and s_pdu_eop='0' and streaming_mode='0' then
            if v_in_frame='0' then 
                -- synthesis translate_off
                report "[pdu_chk]: data received out of frame." severity warning;
                s_stop_sim <= '1';
                -- synthesis translate_on
                s_error <= '1';
            end if;
        end if;
        
        -- Check EOP is received in frame
        if s_pdu_vld='1' and s_pdu_eop='1' then
            if v_in_frame='0' then 
                -- synthesis translate_off
                report "[pdu_chk]: EOP received out of frame." severity warning;
                s_stop_sim <= '1';
                -- synthesis translate_on
                s_error <= '1';
            else
                v_in_frame := '0';
            end if;
        end if;
        s_in_frame <= v_in_frame;
        
        -- Check EOP versus SOP counters
        -- synthesis translate_off
        if unsigned(s_eop_cnt)>unsigned(s_sop_cnt) then 
            report "[pdu_chk]: too many EOP received compared to SOP." severity warning;
            s_stop_sim <= '1';
        end if;
        if unsigned(s_eop_cnt)+1<unsigned(s_sop_cnt) then 
            report "[pdu_chk]: too many SOP received compared to EOP." severity warning;
            s_stop_sim <= '1';
        end if;
        -- synthesis translate_on
        
        ----------------------------------------------------------------------------------------------------
        -- Update counters
        ----------------------------------------------------------------------------------------------------
        -- Frame counter
           if reset_cnt='1' then s_frame_cnt <= (others=>'0');
        elsif streaming_mode = '0' then
           if s_pdu_vld='1' and s_pdu_eop='1' then s_frame_cnt <= s_frame_cnt + 1; end if; 
        else -- streaming_mode = '1' : makes a pseudo Frame Counter every time the Data make a rollover to 0x0100 
           if s_pdu_vld='1' and s_pdu_d(15 downto 0) = x"0100" then s_frame_cnt <= s_frame_cnt + 1; end if;
        end if;
        
        -- Error counter
           if reset_cnt='1' then s_error_cnt <= (others=>'0');
        elsif s_error='1'   then s_error_cnt <= s_error_cnt + 1; end if;            
        
        -- Error Link counter
           if reset_cnt='1'    then s_error_link_cnt <= (others=>'0');
        elsif s_error_link='1' then s_error_link_cnt <= s_error_link_cnt + 1; end if;            
        
    end if;
    end process p_main;
    
    ----------------------------------------------------------------------------------------------------
    -- synthesis translate_off
    process begin
        wait until rising_edge(s_stop_sim);
        
        wait for 100 ns;
        report "[pdu_chk]: stopping simulation because of error." severity failure;
    end process;
    -- synthesis translate_on
    ----------------------------------------------------------------------------------------------------
    
    --##############################################################
    -- Internal / Ports assignments
    --############################################################## 
    s_pdu_d        <= pdu_d;
    s_pdu_vld      <= pdu_vld;
    s_pdu_bena     <= pdu_bena;
    s_pdu_sop      <= pdu_sop;
    s_pdu_eop      <= pdu_eop;
    s_pdu_err      <= pdu_err;
    frame_cnt      <= s_frame_cnt;
    error_cnt      <= s_error_cnt;
    error_link_cnt <= s_error_link_cnt;
    
end architecture rtl;
