----------------------------------------------------------------------------------------------------
-- Copyright (c) ReFLEX CES 1998-2012
--
-- Use of this source code through a simulator and/or a compiler tool
-- is illegal if not authorised through ReFLEX CES License agreement.
----------------------------------------------------------------------------------------------------
-- Project     : 
----------------------------------------------------------------------------------------------------
-- Top level   : top.vhd
-- File        : nfc_gen.vhd
-- Author      : flavenant@reflexces.com
-- Company     : ReFLEX CES
--               2, rue du gevaudan
--               91047 LISSES
--               FRANCE
--               http://www.reflexces.com
-- Plateforme  : Windows XP
-- Simulator   : Mentor Graphics ModelSim
-- Synthesis   : Quartus II
-- Target      : Stratix IV
-- Dependency  :
----------------------------------------------------------------------------------------------------
-- Description :
-- NFC generator
--  
----------------------------------------------------------------------------------------------------
-- Version      Date            Author              Description
-- 0.1          2012/01/19      FLA                 Creation
----------------------------------------------------------------------------------------------------
library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;
use     ieee.std_logic_unsigned.all;

entity nfc_gen is
    generic (
          COUNTER_WIDTH         : integer := 32         -- Number of bits for the frame counter
        ; AUTO_GEN              : boolean := true       -- If true, NFC are requested automatically, if false nfc_start must be asserted by the user.
        ; AUTO_GEN_BITS         : integer := 8          -- Number of bits for the auto gen free run counter (NFC is sent when counter=0 and LFSR=1);
    );
    port (
        -- Reset and clocks
          rst_n                 : in    std_logic  
        ; clk                   : in    std_logic  
        
        -- NFC
        ; nfc_d                 : out   std_logic_vector(3 downto 0)
        ; nfc_vld               : out   std_logic
        ; nfc_rdy               : in    std_logic
        
        -- Control / Status
        ; enable                : in    std_logic                                       
        ; nfc_start             : in    std_logic                    := '0'             -- one cycle pulse to send an NFC.
        ; nfc_start_nb          : in    std_logic_vector(3 downto 0) := (others=>'0')   -- number of pause to generate
        ; wait_cnt              : out   std_logic_vector(COUNTER_WIDTH-1 downto 0)      -- add nfc_d+1 on every nfc sent
        ; reset_cnt             : in    std_logic
    );
end entity nfc_gen;

architecture rtl of nfc_gen is
	----------------------------------------------------------------
	-- Type declarations
	----------------------------------------------------------------
    type TState is (FS_REQ, FS_ACK);
    
	----------------------------------------------------------------
	-- Function declarations
	----------------------------------------------------------------
        
    -- LFSR (Linear Feedback Shift Register) function for test
    -- /!\ use xnor to set 111...11 as the idle state instead of 000...00
    -- '1' bit in mask enables xnor with corresponding bit in vectIn
    function DoLFSR(vectIn : std_logic_vector;
                    mask   : std_logic_vector ) return std_logic_vector is
        variable v_rtn : std_logic_vector(vectIn'range);
        variable v_fb  : std_logic;
    begin
        v_fb              := vectIn(0);
        v_rtn(v_rtn'high) := v_fb;
    
        for i in vectIn'low to vectIn'high-1 loop
			if (mask(i+1)='1') then v_rtn(i) := vectIn(i+1) xnor v_fb;
			else                    v_rtn(i) := vectIn(i+1);
			end if;
        end loop;
        
        return v_rtn;
    end function DoLFSR;
    
	----------------------------------------------------------------
	-- Component declarations
	----------------------------------------------------------------
    
	----------------------------------------------------------------
	-- Constant declarations
	----------------------------------------------------------------
    constant C_LFSR_REQ                 : std_logic_vector(28 downto 0) := std_logic_vector(to_unsigned(16#4f3#, 29)); 
    constant C_LFSR_NB                  : std_logic_vector(28 downto 0) := std_logic_vector(to_unsigned(16#4f3#, 29)); 

	----------------------------------------------------------------
	-- Signal declarations
	----------------------------------------------------------------
    signal s_state              : TState;
    signal s_do_req             : std_logic;
    signal s_nfc_req            : std_logic;
    signal s_nfc_ack            : std_logic;
    signal s_nfc_nb             : std_logic_vector(nfc_d'range);
    signal s_req_nb             : std_logic_vector(nfc_d'range);
    signal s_tempo_cnt          : std_logic_vector(AUTO_GEN_BITS-1 downto 0);
    signal s_tempo_cnt0         : std_logic;
    signal s_lfsr_req           : std_logic_vector(C_LFSR_REQ'range);
    signal s_lfsr_nb            : std_logic_vector(C_LFSR_NB'range);
    signal s_wait_cnt           : std_logic_vector(COUNTER_WIDTH-1 downto 0);
    signal s_xoff               : std_logic;
begin

    --##############################################################
    -- Main process
    --##############################################################     
    p_main : process (clk, rst_n)
        variable v_do_req           : std_logic;
        variable v_req_nb           : std_logic_vector(s_req_nb'range);
    begin
    if rst_n='0' then 
        s_lfsr_req          <= C_LFSR_REQ;
        s_lfsr_nb           <= C_LFSR_NB;
        s_state             <= FS_REQ;
        s_tempo_cnt         <= (others=>'0');
        s_tempo_cnt0        <= '0';
        s_do_req            <= '0';
        s_xoff              <= '0';
        s_nfc_req           <= '0';
        s_nfc_nb            <= (others=>'0');
        s_req_nb            <= (others=>'0');
        s_wait_cnt          <= (others=>'0');
        
    elsif rising_edge(clk) then
    
        ----------------------------------------------------------------------------------------------------
        -- Manage frame counter
        ----------------------------------------------------------------------------------------------------
        -- Count frames sent
           if reset_cnt='1'                    then s_wait_cnt <= (others=>'0');
        elsif s_state=FS_ACK and s_nfc_ack='1' then s_wait_cnt <= s_wait_cnt + s_req_nb + 1; end if;
    
        ----------------------------------------------------------------------------------------------------
        -- NFC requester
        ----------------------------------------------------------------------------------------------------
        -- Assert this variable when a request is to be sent
           if AUTO_GEN=true  and s_state=FS_REQ then v_do_req := s_lfsr_req(0) and s_tempo_cnt0 and (enable or s_xoff);
        elsif AUTO_GEN=false and s_state=FS_REQ then v_do_req := nfc_start     and                   enable           ;
        else                                         v_do_req := '0'; end if;
        
        -- Free counter
        s_tempo_cnt <= s_tempo_cnt + 1;
        if s_tempo_cnt=0 then s_tempo_cnt0 <= '1';
        else                  s_tempo_cnt0 <= '0'; end if;
        
        -- LFSR for random requests
        if s_tempo_cnt0='1' then s_lfsr_req <= DoLFSR(s_lfsr_req, C_LFSR_REQ); end if;
        
        -- NFC request
           if s_state=FS_REQ and v_do_req='1' then s_do_req <= '1';
        elsif s_state=FS_ACK                  then s_do_req <= '0'; end if;
        
        -- NFC request NB (only allow XON after a XOFF)
        s_lfsr_nb <= DoLFSR(s_lfsr_nb, C_LFSR_NB);
        v_req_nb  := s_lfsr_nb(v_req_nb'range);
        if v_req_nb(v_req_nb'high)='1' then v_req_nb(2 downto 0) := (others=>v_req_nb(0)); end if; -- change reserved codes

           if v_do_req='1' and s_xoff='0' and nfc_start='0' then s_req_nb <= v_req_nb;
        elsif v_do_req='1' and s_xoff='0' and nfc_start='1' then s_req_nb <= nfc_start_nb;
        elsif v_do_req='1' and s_xoff='1'                   then s_req_nb <= (others=>'0'); end if;
        
        -- Hold XOFF request
           if s_nfc_ack='1' and s_req_nb ="1111" then s_xoff <= '1';
        elsif s_nfc_ack='1' and s_req_nb/="1111" then s_xoff <= '0'; end if;
        
        ----------------------------------------------------------------------------------------------------
        -- Manage FSM
        ----------------------------------------------------------------------------------------------------
        -- FSM
        case s_state is
            when FS_REQ  => if s_do_req='1'  then s_state <= FS_ACK; end if;
            when FS_ACK  => if s_nfc_ack='1' then s_state <= FS_REQ; end if;
        end case;
        
        ----------------------------------------------------------------------------------------------------
        -- NFC bus
        ----------------------------------------------------------------------------------------------------
        -- Build 'req' output
           if s_state=FS_REQ and s_do_req='1'  then s_nfc_req <= '1';
        elsif s_state=FS_ACK and s_nfc_ack='1' then s_nfc_req <= '0'; end if;
        
        -- Build 'nb' output
        if s_state=FS_REQ and s_do_req='1' then s_nfc_nb  <= s_req_nb; end if;
        
    end if;
    end process p_main;
    
    --##############################################################
    -- Internal / Ports assignments
    --##############################################################  
    s_nfc_ack <= nfc_rdy;
    nfc_vld   <= s_nfc_req;
    nfc_d     <= s_nfc_nb;    
    wait_cnt   <= s_wait_cnt;
    
end architecture rtl;
