-- ----------------------------------------------------------------
--  Copyright    : ALSE - http://alse-fr.com
--  Contact      : info@alse-fr.com
--  Project Name : ALSE - Aurora IP for Altera Devices
--  Block Name   : xil_aurora_tester_loopback
--  Description  : Xilinx Aurora IP Tester Master
--  Author       : E.LAURENDEAU
--  Creation     : 05/07/2018
-- ----------------------------------------------------------------

library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;

Library UNISIM;
use UNISIM.vcomponents.all;

-----------------------------------------------------
entity xil_aurora_tester_master is
-----------------------------------------------------
  generic (
    nb_lanes       : integer   :=  1;  -- Number of XCVR Lanes
    lane_width     : integer   := 16;  -- 16 or 32
    streaming_mode : std_logic := '0'; -- Streaming Mode if '1', Framing if '0'
    simplex_rx     : std_logic := '0'; -- Simplex Rx only if '1'
    crc_enable     : std_logic := '0'; -- CRC Enabled (only Framing Mode)
    frame_cnt_bits : integer   := 32
  );
  port (

    -- Reset and clocks
    rst_n      : in  std_logic; -- asynchronous active low reset
    init_clk   : in  std_logic; -- clock for initialization
    gt_clk_p   : in  std_logic; -- clock for GT
    gt_clk_n   : in  std_logic; -- clock for GT
    gt_clk_o   : out std_logic; -- clock for GT
    user_clk   : out std_logic; -- Aurora User Clock

    -- Control Signals
    pdu_tx_ena       : in std_logic;        -- Set to '1' to Enable PDU Sending by Xilinx
    pdu_underflow    : in std_logic;        -- '1' to generate PDU not valid cycles
    pdu_endless_mode : in std_logic := '0'; -- '1' to generate PDU endless PDU frames (no EOP)
    nfc_tx_ena       : in std_logic := '0'; -- Set to '1' to Enable NFC Sending by Xilinx
    ufc_tx_ena       : in std_logic;        -- Set to '1' to Enable UFC Sending by Xilinx
    reset_cnt        : in std_logic := '0'; -- Reset all counters to 0

    -- PDU Fixed Mode
    pdu_fixed_size_on    : in  std_logic;                     -- Fixed Frame Mode if '1'
    pdu_fixed_size_max   : in  std_logic_vector(31 downto 0); -- Fixed Frame Mode Max Size
    pdu_fixed_size_min   : in  std_logic_vector(31 downto 0); -- Fixed Frame Mode Min Size
    pdu_fixed_size_incr  : in  std_logic_vector(31 downto 0); -- Fixed Frame Mode Size Increment
    pdu_fixed_size_ifgap : in  std_logic_vector(31 downto 0); -- Fixed Frame Mode InterFrame Gap
    pdu_fixed_cut_size   : in  std_logic_vector(31 downto 0); -- Fixed Frame Cut Size
    pdu_fixed_cut_gap    : in  std_logic_vector(31 downto 0); -- Fixed Frame Cut Gap

    -- XCVR Lanes
    xcvr_rx_p  : in  std_logic_vector(nb_lanes-1 downto 0);
    xcvr_rx_n  : in  std_logic_vector(nb_lanes-1 downto 0);
    xcvr_tx_p  : out std_logic_vector(nb_lanes-1 downto 0);
    xcvr_tx_n  : out std_logic_vector(nb_lanes-1 downto 0);

    -- Status
    link_up    : out std_logic;
    lane_up    : out std_logic_vector(nb_lanes-1 downto 0);
    status     : out std_logic_vector(15 downto 0);
    pdu_tx_frame_size : out std_logic_vector(31 downto 0);
    pdu_tx_frame_cnt  : out std_logic_vector(frame_cnt_bits-1 downto 0);
    pdu_rx_frame_cnt  : out std_logic_vector(frame_cnt_bits-1 downto 0);
    pdu_rx_error_cnt  : out std_logic_vector(frame_cnt_bits-1 downto 0);
    ufc_rx_frame_cnt  : out std_logic_vector(frame_cnt_bits-1 downto 0);
    ufc_rx_error_cnt  : out std_logic_vector(frame_cnt_bits-1 downto 0);
    nfc_rx_wait_cnt   : out std_logic_vector(frame_cnt_bits-1 downto 0)

  );
-----------------------------------------------------
end entity xil_aurora_tester_master;
-----------------------------------------------------

-----------------------------------------------------
architecture rtl of xil_aurora_tester_master is
-----------------------------------------------------

  ----------------------------------------------------------------
  -- Constant declarations
  ----------------------------------------------------------------
  constant c_nb_16b_lanes   : integer := lane_width/16*nb_lanes;

  ----------------------------------------------------------------
  -- Function declarations
  ----------------------------------------------------------------

  ----------------------------------------------------------------
  -- Component declarations
  ----------------------------------------------------------------
  -- Clock  Transceiver Buffer
  component IBUFDS_GTE2
    port (
      O            : out std_ulogic;
      ODIV2        : out std_ulogic;
      CEB          : in std_ulogic;
      I            : in std_ulogic;
      IB           : in std_ulogic
    );
  end component;

  -- Clock Buffer
  component BUFG
    port (
      O            : out std_ulogic;
      I            : in std_ulogic
    );
  end component;

  -- GTX Channel for specific clocking scheme
  component GTXE2_COMMON
    generic (
      SIM_QPLLREFCLK_SEL         : bit_vector := "101"              ;
      SIM_RESET_SPEEDUP          : string     := "False"            ;
      SIM_VERSION                : string     := "1.0"              ;

      BIAS_CFG                   : bit_vector := x"0000040000001000";
      COMMON_CFG                 : bit_vector := x"00000000"        ;
      QPLL_CFG                   : bit_vector := x"06801C1"         ;
      QPLL_CLKOUT_CFG            : bit_vector := "0000"             ;
      QPLL_COARSE_FREQ_OVRD      : bit_vector := "010000"           ;
      QPLL_COARSE_FREQ_OVRD_EN   : bit        := '0'                ;
      QPLL_CP                    : bit_vector := "0000011111"       ;
      QPLL_CP_MONITOR_EN         : bit        := '0'                ;
      QPLL_DMONITOR_SEL          : bit        := '0'                ;
      QPLL_FBDIV                 : bit_vector := "0010000000"       ;
      QPLL_FBDIV_MONITOR_EN      : bit        := '0'                ;
      QPLL_FBDIV_RATIO           : bit        := '1'                ;
      QPLL_INIT_CFG              : bit_vector := x"000006"          ;
      QPLL_LOCK_CFG              : bit_vector := x"21E8"            ;
      QPLL_LPF                   : bit_vector := "1111"             ;
      QPLL_REFCLK_DIV            : integer    := 1
    );
    port (
      DRPDO                      : out std_logic_vector(15 downto 0);
      DRPRDY                     : out std_ulogic;
      QPLLDMONITOR               : out std_logic_vector(7 downto 0);
      QPLLFBCLKLOST              : out std_ulogic;
      QPLLLOCK                   : out std_ulogic;
      QPLLOUTCLK                 : out std_ulogic;
      QPLLOUTREFCLK              : out std_ulogic;
      QPLLREFCLKLOST             : out std_ulogic;
      REFCLKOUTMONITOR           : out std_ulogic;
      BGBYPASSB                  : in std_ulogic;
      BGMONITORENB               : in std_ulogic;
      BGPDB                      : in std_ulogic;
      BGRCALOVRD                 : in std_logic_vector(4 downto 0);
      DRPADDR                    : in std_logic_vector(7 downto 0);
      DRPCLK                     : in std_ulogic;
      DRPDI                      : in std_logic_vector(15 downto 0);
      DRPEN                      : in std_ulogic;
      DRPWE                      : in std_ulogic;
      GTGREFCLK                  : in std_ulogic;
      GTNORTHREFCLK0             : in std_ulogic;
      GTNORTHREFCLK1             : in std_ulogic;
      GTREFCLK0                  : in std_ulogic;
      GTREFCLK1                  : in std_ulogic;
      GTSOUTHREFCLK0             : in std_ulogic;
      GTSOUTHREFCLK1             : in std_ulogic;
      PMARSVD                    : in std_logic_vector(7 downto 0);
      QPLLLOCKDETCLK             : in std_ulogic;
      QPLLLOCKEN                 : in std_ulogic;
      QPLLOUTRESET               : in std_ulogic;
      QPLLPD                     : in std_ulogic;
      QPLLREFCLKSEL              : in std_logic_vector(2 downto 0);
      QPLLRESET                  : in std_ulogic;
      QPLLRSVD1                  : in std_logic_vector(15 downto 0);
      QPLLRSVD2                  : in std_logic_vector(4 downto 0);
      RCALENB                    : in std_ulogic
    );
  end component;

  -- Xilinx Aurora IP - x1 - 16bits - Framing Mode
  component aurora_8b10b_x1_16b is
    port (
      -- AXI TX Interface
      s_axi_tx_tready         : out std_logic;
      s_axi_tx_tvalid         : in  std_logic;
      s_axi_tx_tdata          : in  std_logic_vector(15 downto 0);
      s_axi_tx_tkeep          : in  std_logic_vector( 1 downto 0);
      s_axi_tx_tlast          : in  std_logic;
      -- AXI RX Interface
      m_axi_rx_tvalid         : out std_logic;
      m_axi_rx_tdata          : out std_logic_vector(15 downto 0);
      m_axi_rx_tkeep          : out std_logic_vector( 1 downto 0);
      m_axi_rx_tlast          : out std_logic;
      -- Native Flow Control TX Interface
      s_axi_nfc_tx_tvalid     : in  std_logic;
      s_axi_nfc_tx_tdata      : in  std_logic_vector( 3 downto 0);
      s_axi_nfc_tx_tready     : out std_logic;
      -- Native Flow Control RX Interface
      m_axi_nfc_rx_tvalid     : out std_logic;
      m_axi_nfc_rx_tdata      : out std_logic_vector( 3 downto 0);
      -- User Flow Control TX Interface
      s_axi_ufc_tx_tvalid     : in  std_logic;
      s_axi_ufc_tx_tdata      : in  std_logic_vector( 2 downto 0);
      s_axi_ufc_tx_tready     : out std_logic;
      -- User Flow Control RX Inteface
      m_axi_ufc_rx_tdata      : out std_logic_vector(15 downto 0);
      m_axi_ufc_rx_tkeep      : out std_logic_vector( 1 downto 0);
      m_axi_ufc_rx_tvalid     : out std_logic;
      m_axi_ufc_rx_tlast      : out std_logic;
      -- GT Serial I/O
      rxp                     : in  std_logic_vector(0 downto 0);
      rxn                     : in  std_logic_vector(0 downto 0);
      txp                     : out std_logic_vector(0 downto 0);
      txn                     : out std_logic_vector(0 downto 0);
      -- GT Reference Clock Interface
      gt_refclk1              : in  std_logic;
      -- Error Detection Interface
      frame_err               : out std_logic;
      hard_err                : out std_logic;
      soft_err                : out std_logic;
      channel_up              : out std_logic;
      lane_up                 : out std_logic_vector(0 downto 0);
      -- System Interface
      reset                   : in  std_logic;
      power_down              : in  std_logic;
      loopback                : in  std_logic_vector( 2 downto 0);
      tx_lock                 : out std_logic;
      sys_reset_out           : out std_logic;
      gt_reset                : in  std_logic;
      init_clk_in             : in  std_logic;
      tx_resetdone_out        : out std_logic;
      rx_resetdone_out        : out std_logic;
      link_reset_out          : out std_logic;
      --DRP Ports
      drpclk_in               : in  std_logic;
      drpaddr_in              : in  std_logic_vector( 8 downto 0);
      drpdi_in                : in  std_logic_vector(15 downto 0);
      drpdo_out               : out std_logic_vector(15 downto 0);
      drpen_in                : in  std_logic;
      drprdy_out              : out std_logic;
      drpwe_in                : in  std_logic;
      -- Clocking
      tx_out_clk              : out std_logic;
      user_clk                : in std_logic;
      sync_clk                : in std_logic;
      --
      gt0_qplllock_in         : in std_logic;
      gt0_qpllrefclklost_in   : in std_logic;
      gt0_qpllreset_out       : out std_logic;
      gt_qpllclk_quad1_in     : in std_logic;
      gt_qpllrefclk_quad1_in  : in std_logic;
      pll_not_locked          : in std_logic
    );
  end component aurora_8b10b_x1_16b;

  -- Xilinx Aurora IP - x1 - 32bits - Framing Mode
  component aurora_8b10b_x1_32b is
    port (
      -- AXI TX Interface
      s_axi_tx_tready         : out std_logic;
      s_axi_tx_tvalid         : in  std_logic;
      s_axi_tx_tdata          : in  std_logic_vector(31 downto 0);
      s_axi_tx_tkeep          : in  std_logic_vector( 3 downto 0);
      s_axi_tx_tlast          : in  std_logic;
      -- AXI RX Interface
      m_axi_rx_tvalid         : out std_logic;
      m_axi_rx_tdata          : out std_logic_vector(31 downto 0);
      m_axi_rx_tkeep          : out std_logic_vector( 3 downto 0);
      m_axi_rx_tlast          : out std_logic;
      -- Native Flow Control TX Interface
      s_axi_nfc_tx_tvalid     : in  std_logic;
      s_axi_nfc_tx_tdata      : in  std_logic_vector( 3 downto 0);
      s_axi_nfc_tx_tready     : out std_logic;
      -- Native Flow Control RX Interface
      m_axi_nfc_rx_tvalid     : out std_logic;
      m_axi_nfc_rx_tdata      : out std_logic_vector( 3 downto 0);
      -- User Flow Control TX Interface
      s_axi_ufc_tx_tvalid     : in  std_logic;
      s_axi_ufc_tx_tdata      : in  std_logic_vector( 2 downto 0);
      s_axi_ufc_tx_tready     : out std_logic;
      -- User Flow Control RX Inteface
      m_axi_ufc_rx_tdata      : out std_logic_vector(31 downto 0);
      m_axi_ufc_rx_tkeep      : out std_logic_vector( 3 downto 0);
      m_axi_ufc_rx_tvalid     : out std_logic;
      m_axi_ufc_rx_tlast      : out std_logic;
      -- GT Serial I/O
      rxp                     : in  std_logic_vector(0 downto 0);
      rxn                     : in  std_logic_vector(0 downto 0);
      txp                     : out std_logic_vector(0 downto 0);
      txn                     : out std_logic_vector(0 downto 0);
      -- GT Reference Clock Interface
      gt_refclk1              : in  std_logic;
      -- Error Detection Interface
      frame_err               : out std_logic;
      hard_err                : out std_logic;
      soft_err                : out std_logic;
      channel_up              : out std_logic;
      lane_up                 : out std_logic_vector(0 downto 0);
      -- System Interface
      reset                   : in  std_logic;
      power_down              : in  std_logic;
      loopback                : in  std_logic_vector( 2 downto 0);
      gt_reset                : in  std_logic;
      tx_lock                 : out std_logic;
      sys_reset_out           : out std_logic;
      init_clk_in             : in  std_logic;
      tx_resetdone_out        : out std_logic;
      rx_resetdone_out        : out std_logic;
      link_reset_out          : out std_logic;
      --DRP Ports
      drpclk_in               : in  std_logic;
      drpaddr_in              : in  std_logic_vector( 8 downto 0);
      drpdi_in                : in  std_logic_vector(15 downto 0);
      drpdo_out               : out std_logic_vector(15 downto 0);
      drpen_in                : in  std_logic;
      drprdy_out              : out std_logic;
      drpwe_in                : in  std_logic;
      -- Clocking
      tx_out_clk              : out std_logic;
      user_clk                : in std_logic;
      sync_clk                : in std_logic;
      --
      gt0_qplllock_in         : in std_logic;
      gt0_qpllrefclklost_in   : in std_logic;
      gt0_qpllreset_out       : out std_logic;
      gt_qpllclk_quad1_in     : in std_logic;
      gt_qpllrefclk_quad1_in  : in std_logic;
      pll_not_locked          : in std_logic
    );
  end component aurora_8b10b_x1_32b;

  -- Xilinx Aurora IP - x2 - 16bits - Framing Mode
  component aurora_8b10b_x2_16b
  port (
    -- AXI TX Interface
    s_axi_tx_tdata         : in  std_logic_vector(31 downto 0);
    s_axi_tx_tkeep         : in  std_logic_vector(3 downto 0);
    s_axi_tx_tlast         : in  std_logic;
    s_axi_tx_tvalid        : in  std_logic;
    s_axi_tx_tready        : out std_logic;
    -- AXI RX Interface
    m_axi_rx_tdata         : out std_logic_vector(31 downto 0);
    m_axi_rx_tkeep         : out std_logic_vector(3 downto 0);
    m_axi_rx_tlast         : out std_logic;
    m_axi_rx_tvalid        : out std_logic;
    -- Native Flow Control TX Interface
    s_axi_nfc_tx_tvalid    : in  std_logic;
    s_axi_nfc_tx_tdata     : in  std_logic_vector(3 downto 0);
    s_axi_nfc_tx_tready    : out std_logic;
    -- Native Flow Control RX Interface
    m_axi_nfc_rx_tvalid    : out std_logic;
    m_axi_nfc_rx_tdata     : out std_logic_vector(3 downto 0);
    -- User Flow Control TX Interface
    s_axi_ufc_tx_tvalid    : in  std_logic;
    s_axi_ufc_tx_tdata     : in  std_logic_vector(2 downto 0);
    s_axi_ufc_tx_tready    : out std_logic;
    -- User Flow Control RX Inteface
    m_axi_ufc_rx_tdata     : out std_logic_vector(31 downto 0);
    m_axi_ufc_rx_tkeep     : out std_logic_vector(3 downto 0);
    m_axi_ufc_rx_tlast     : out std_logic;
    m_axi_ufc_rx_tvalid    : out std_logic;
    -- GT Serial I/O
    txp                    : out std_logic_vector(0 to 1);
    txn                    : out std_logic_vector(0 to 1);
    rxp                    : in  std_logic_vector(0 to 1);
    rxn                    : in  std_logic_vector(0 to 1);
    -- GT Reference Clock Interface
    gt_refclk1             : in  std_logic;
    -- Error Detection Interface
    frame_err              : out std_logic;
    hard_err               : out std_logic;
    soft_err               : out std_logic;
    channel_up             : out std_logic;
    lane_up                : out std_logic_vector(0 to 1);
    -- System Interface
    reset                  : in  std_logic;
    power_down             : in  std_logic;
    loopback               : in  std_logic_vector(2 downto 0);
    gt_reset               : in  std_logic;
    tx_lock                : out std_logic;
    sys_reset_out          : out std_logic;
    init_clk_in            : in  std_logic;
    tx_resetdone_out       : out std_logic;
    rx_resetdone_out       : out std_logic;
    link_reset_out         : out std_logic;
    --DRP Ports
    drpclk_in              : in  std_logic;
    drpaddr_in             : in  std_logic_vector(8 downto 0);
    drpen_in               : in  std_logic;
    drpdi_in               : in  std_logic_vector(15 downto 0);
    drprdy_out             : out std_logic;
    drpdo_out              : out std_logic_vector(15 downto 0);
    drpwe_in               : in  std_logic;
    drpaddr_in_lane1       : in  std_logic_vector(8 downto 0);
    drpen_in_lane1         : in  std_logic;
    drpdi_in_lane1         : in  std_logic_vector(15 downto 0);
    drprdy_out_lane1       : out std_logic;
    drpdo_out_lane1        : out std_logic_vector(15 downto 0);
    drpwe_in_lane1         : in  std_logic;
    -- Clocking
    tx_out_clk             : out std_logic;
    user_clk               : in  std_logic;
    sync_clk               : in  std_logic;
    --
    gt0_qplllock_in        : in  std_logic;
    gt0_qpllrefclklost_in  : in  std_logic;
    gt0_qpllreset_out      : out std_logic;
    gt_qpllclk_quad1_in    : in  std_logic;
    gt_qpllrefclk_quad1_in : in  std_logic;
    pll_not_locked         : in  std_logic
  );
end component aurora_8b10b_x2_16b;

  ----------------------------------------------------------------
  -- Signal declarations
  ----------------------------------------------------------------

  -- PDU Gen Signals
  signal s_pdu_tx_rdy        : std_logic;
  signal s_pdu_tx_sop        : std_logic;
  signal s_pdu_tx_vld        : std_logic;
  signal s_pdu_tx_d          : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_pdu_tx_bena       : std_logic_vector(lane_width/8*nb_lanes-1 downto 0);
  signal s_pdu_tx_eop        : std_logic;
  signal s_pdu_tx_frame_cnt  : std_logic_vector(frame_cnt_bits-1 downto 0);
  signal s_pdu_tx_frame_size : std_logic_vector(31 downto 0);

  -- PDU Tx Signals
  signal s_pdu_tx_tready : std_logic;
  signal s_pdu_tx_tvalid : std_logic;
  signal s_pdu_tx_tdata  : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_pdu_tx_tkeep  : std_logic_vector(lane_width/8*nb_lanes-1 downto 0);
  signal s_pdu_tx_tlast  : std_logic;
  signal s_pdu_tx_tdata2 : std_logic_vector(lane_width  *nb_lanes-1 downto 0);

  -- PDU Rx Signals
  signal s_pdu_rx_tvalid : std_logic;
  signal s_pdu_rx_tdata  : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_pdu_rx_tkeep  : std_logic_vector(lane_width/8*nb_lanes-1 downto 0);
  signal s_pdu_rx_tlast  : std_logic;

  -- PDU Check Signals
  signal s_pdu_rx_sop       : std_logic;
  signal s_pdu_rx_vld       : std_logic;
  signal s_pdu_rx_d         : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_pdu_rx_bena      : std_logic_vector(lane_width/8*nb_lanes-1 downto 0);
  signal s_pdu_rx_eop       : std_logic;
  signal s_pdu_rx_frame_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);
  signal s_pdu_rx_error_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);

  -- UFC Tx Signals
  signal s_ufc_tx_tready    : std_logic;
  signal s_ufc_tx_tvalid    : std_logic;
  signal s_ufc_tx_tdata     : std_logic_vector(2 downto 0);
  signal s_ufc_tx_data_f    : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_ufc_tx_frame_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);

  -- UFC Rx Signals
  signal s_ufc_rx_tvalid    : std_logic;
  signal s_ufc_rx_tdata     : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_ufc_rx_tkeep     : std_logic_vector(lane_width/8*nb_lanes-1 downto 0);
  signal s_ufc_rx_tlast     : std_logic;
  signal s_ufc_rx_sop       : std_logic;
  signal s_ufc_rx_vld       : std_logic;
  signal s_ufc_rx_d         : std_logic_vector(lane_width  *nb_lanes-1 downto 0);
  signal s_ufc_rx_bena      : std_logic_vector(lane_width/8*nb_lanes-1 downto 0);
  signal s_ufc_rx_eop       : std_logic;
  signal s_ufc_rx_frame_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);
  signal s_ufc_rx_error_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);
  signal s_ufc_rx_frame_ok  : std_logic;
  signal s_ufc_rx_frame_err : std_logic;

  -- NFC Tx Signals
  signal s_nfc_tx_tready   : std_logic;
  signal s_nfc_tx_tvalid   : std_logic;
  signal s_nfc_tx_tdata    : std_logic_vector(3 downto 0);
  signal s_nfc_tx_wait_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);

  -- NFC Rx Signals
  signal s_nfc_rx_tvalid   : std_logic;
  signal s_nfc_rx_tdata    : std_logic_vector(3 downto 0);
  signal s_nfc_rx_wait_cnt : std_logic_vector(frame_cnt_bits-1 downto 0);

  -- Clocking
  signal tx_out_clk : std_logic; -- user clock generated by transceiver
  signal s_user_clk : std_logic;
  signal s_sync_clk : std_logic;
  signal gt_clk     : std_logic;
  signal pll_not_locked : std_logic;

  -- Common Block - QPLL Ports
  signal gt0_qplllock_i        : std_logic;
  signal gt0_qpllrefclklost_i  : std_logic;
  signal gt0_qpllreset_i       : std_logic;
  signal gt_qpllclk_quad1_i    : std_logic;
  signal gt_qpllrefclk_quad1_i : std_logic;

  -- Other XCVR signals
  signal s_hard_err   : std_logic;
  signal s_soft_err   : std_logic;
  signal s_frame_err  : std_logic;
  signal s_channel_up : std_logic;
  signal s_lane_up    : std_logic_vector(nb_lanes-1 downto 0);
  signal s_tx_lock    : std_logic;

  -- CRC
  signal crc_pass_fail_n : std_logic;
  signal crc_valid       : std_logic;

  -- Misc
  signal s_rst : std_logic;



--------
begin
--------


  -- -------------------------------
  -- Signal assignments
  -- -------------------------------
  s_rst   <= not rst_n;
  link_up <= s_channel_up;
  lane_up <= s_lane_up;

  -- ---------------------- --
  -- GT differential buffer --
  -- ---------------------- --
  IBUFDS_GTE2_i : IBUFDS_GTE2
    port map (
      O     => gt_clk,   -- 1-bit output: Refer to Transceiver User Guide
      ODIV2 => open,     -- 1-bit output: Refer to Transceiver User Guide
      CEB   => '0',        -- 1-bit input: Refer to Transceiver User Guide
      I     => gt_clk_p, -- 1-bit input: Refer to Transceiver User Guide
      IB    => gt_clk_n -- 1-bit input: Refer to Transceiver User Guide
  );

  -- ---------------------- --
  -- GT Common Clock Module --
  -- ---------------------- --
  -- Taken from example project
  GTXE2_COMMON_i : GTXE2_COMMON
    port map (
      DRPADDR                    => x"00"                ,
      DRPCLK                     => '0'                  ,
      DRPDI                      => x"0000"              ,
      DRPDO                      => open                 ,
      DRPEN                      => '0'                  ,
      DRPRDY                     => open                 ,
      DRPWE                      => '0'                  ,
      --
      GTGREFCLK                  => '0'                  ,
      GTNORTHREFCLK0             => '0'                  ,
      GTNORTHREFCLK1             => '0'                  ,
      GTREFCLK0                  => '0'                  ,
      GTREFCLK1                  => '0'                  ,
      GTSOUTHREFCLK0             => gt_clk               ,
      GTSOUTHREFCLK1             => '0'                  ,
      --
      QPLLDMONITOR               => open                 ,
      QPLLFBCLKLOST              => open                 ,
      QPLLLOCK                   => gt0_qplllock_i       ,
      QPLLLOCKDETCLK             => init_clk             ,
      QPLLLOCKEN                 => '1'                  ,
      QPLLOUTCLK                 => gt_qpllclk_quad1_i   ,
      QPLLOUTREFCLK              => gt_qpllrefclk_quad1_i,
      QPLLOUTRESET               => '0'                  ,
      QPLLPD                     => '1'                  ,
      QPLLREFCLKLOST             => gt0_qpllrefclklost_i ,
      QPLLREFCLKSEL              => "101"                ,
      QPLLRESET                  => gt0_qpllreset_i      ,
      QPLLRSVD1                  => "0000000000000000"   ,
      QPLLRSVD2                  => "11111"              ,
      REFCLKOUTMONITOR           => open                 ,
      --
      BGBYPASSB                  => '1'                  ,
      BGMONITORENB               => '1'                  ,
      BGPDB                      => '1'                  ,
      BGRCALOVRD                 => "11111"              ,
      PMARSVD                    => "00000000"           ,
      RCALENB                    => '1'
  );

  -- -----------------
  -- USER CLK Buffer
  -- -----------------
  user_clk_buf_i : BUFG
    port map (
      I => tx_out_clk,
      O => s_user_clk
  );

  s_sync_clk <= s_user_clk;
  pll_not_locked <= not s_tx_lock;

  -- ----------------------------------------------------
  --  Xilinx Aurora IP Instance, Framing Mode
  -- ----------------------------------------------------
  gen_framing : if streaming_mode = '0' generate

    -- Aurora IP - Framing - x1 - 16b
    gen_x1_16b : if nb_lanes = 1 and lane_width = 16 and crc_enable = '0' generate
      i_aurora_8b10b_x1_16b : aurora_8b10b_x1_16b
        port map   (
          S_AXI_TX_TDATA          => s_pdu_tx_tdata ,
          S_AXI_TX_TKEEP          => s_pdu_tx_tkeep ,
          S_AXI_TX_TVALID         => s_pdu_tx_tvalid,
          S_AXI_TX_TLAST          => s_pdu_tx_tlast ,
          S_AXI_TX_TREADY         => s_pdu_tx_tready,
          M_AXI_RX_TDATA          => s_pdu_rx_tdata ,
          M_AXI_RX_TKEEP          => s_pdu_rx_tkeep ,
          M_AXI_RX_TVALID         => s_pdu_rx_tvalid,
          M_AXI_RX_TLAST          => s_pdu_rx_tlast ,
          S_AXI_NFC_TX_TREADY     => s_nfc_tx_tready,
          S_AXI_NFC_TX_TVALID     => s_nfc_tx_tvalid,
          S_AXI_NFC_TX_TDATA      => s_nfc_tx_tdata ,
          M_AXI_NFC_RX_TVALID     => s_nfc_rx_tvalid,
          M_AXI_NFC_RX_TDATA      => s_nfc_rx_tdata ,
          S_AXI_UFC_TX_TREADY     => s_ufc_tx_tready,
          S_AXI_UFC_TX_TVALID     => s_ufc_tx_tvalid,
          S_AXI_UFC_TX_TDATA      => s_ufc_tx_tdata ,
          M_AXI_UFC_RX_TVALID     => s_ufc_rx_tvalid,
          M_AXI_UFC_RX_TDATA      => s_ufc_rx_tdata ,
          M_AXI_UFC_RX_TKEEP      => s_ufc_rx_tkeep ,
          M_AXI_UFC_RX_TLAST      => s_ufc_rx_tlast ,
          -- XCVR Lanes
          RXP                     => xcvr_rx_p      ,
          RXN                     => xcvr_rx_n      ,
          TXP                     => xcvr_tx_p      ,
          TXN                     => xcvr_tx_n      ,
          -- GT Reference Clock
          gt_refclk1             => gt_clk         ,
          -- Error Detection Interface
          HARD_ERR                => s_hard_err     ,
          SOFT_ERR                => s_soft_err     ,
          FRAME_ERR               => s_frame_err    ,
          CHANNEL_UP              => s_channel_up   ,
          LANE_UP                 => s_lane_up      ,
          --
          reset                   => s_rst          ,
          power_down              => '0'            ,
          loopback                => (others=>'0')  , -- no Loopback of Transceiver
          gt_reset                => s_rst          ,
          tx_lock                 => s_tx_lock      ,
          sys_reset_out           => open           ,
          init_clk_in             => init_clk       ,
          tx_resetdone_out        => open           ,
          rx_resetdone_out        => open           ,
          link_reset_out          => open           ,
          drpclk_in               => '0'            ,
          drpaddr_in              => (others=>'0')  ,
          drpdi_in                => (others=>'0')  ,
          drpdo_out               => open           ,
          drpen_in                => '0'            ,
          drprdy_out              => open           ,
          drpwe_in                => '0'            ,
          --
          user_clk                => s_user_clk            ,
          sync_clk                => s_sync_clk            ,
          tx_out_clk              => tx_out_clk            ,
          gt0_qplllock_in         => gt0_qplllock_i        ,
          gt0_qpllrefclklost_in   => gt0_qpllrefclklost_i  ,
          gt_qpllclk_quad1_in     => gt_qpllclk_quad1_i    ,
          gt_qpllrefclk_quad1_in  => gt_qpllrefclk_quad1_i ,
          gt0_qpllreset_out       => gt0_qpllreset_i       ,
          pll_not_locked          => not s_tx_lock
        );
    end generate gen_x1_16b;

    -- Aurora IP - Framing - x1 - 32b
    gen_x1_32b : if nb_lanes = 1 and lane_width = 32 and crc_enable = '0' generate
      i_aurora_8b10b_x1_32b : aurora_8b10b_x1_32b
        port map   (
          S_AXI_TX_TDATA          => s_pdu_tx_tdata ,
          S_AXI_TX_TKEEP          => s_pdu_tx_tkeep ,
          S_AXI_TX_TVALID         => s_pdu_tx_tvalid,
          S_AXI_TX_TLAST          => s_pdu_tx_tlast ,
          S_AXI_TX_TREADY         => s_pdu_tx_tready,
          M_AXI_RX_TDATA          => s_pdu_rx_tdata ,
          M_AXI_RX_TKEEP          => s_pdu_rx_tkeep ,
          M_AXI_RX_TVALID         => s_pdu_rx_tvalid,
          M_AXI_RX_TLAST          => s_pdu_rx_tlast ,
          S_AXI_NFC_TX_TREADY     => s_nfc_tx_tready,
          S_AXI_NFC_TX_TVALID     => s_nfc_tx_tvalid,
          S_AXI_NFC_TX_TDATA      => s_nfc_tx_tdata ,
          M_AXI_NFC_RX_TVALID     => s_nfc_rx_tvalid,
          M_AXI_NFC_RX_TDATA      => s_nfc_rx_tdata ,
          S_AXI_UFC_TX_TREADY     => s_ufc_tx_tready,
          S_AXI_UFC_TX_TVALID     => s_ufc_tx_tvalid,
          S_AXI_UFC_TX_TDATA      => s_ufc_tx_tdata ,
          M_AXI_UFC_RX_TVALID     => s_ufc_rx_tvalid,
          M_AXI_UFC_RX_TDATA      => s_ufc_rx_tdata ,
          M_AXI_UFC_RX_TKEEP      => s_ufc_rx_tkeep ,
          M_AXI_UFC_RX_TLAST      => s_ufc_rx_tlast ,
          -- XCVR Lanes
          RXP                     => xcvr_rx_p      ,
          RXN                     => xcvr_rx_n      ,
          TXP                     => xcvr_tx_p      ,
          TXN                     => xcvr_tx_n      ,
          -- GT Reference Clock
          gt_refclk1             => gt_clk         ,
          -- Error Detection Interface
          HARD_ERR                => s_hard_err     ,
          SOFT_ERR                => s_soft_err     ,
          FRAME_ERR               => s_frame_err    ,
          CHANNEL_UP              => s_channel_up   ,
          LANE_UP                 => s_lane_up      ,
          --
          reset                   => s_rst          ,
          power_down              => '0'            ,
          loopback                => (others=>'0')  , -- no Loopback of Transceiver
          gt_reset                => s_rst          ,
          tx_lock                 => s_tx_lock      ,
          sys_reset_out           => open           ,
          init_clk_in             => init_clk       ,
          tx_resetdone_out        => open           ,
          rx_resetdone_out        => open           ,
          link_reset_out          => open           ,
          drpclk_in               => '0'            ,
          drpaddr_in              => (others=>'0')  ,
          drpdi_in                => (others=>'0')  ,
          drpdo_out               => open           ,
          drpen_in                => '0'            ,
          drprdy_out              => open           ,
          drpwe_in                => '0'            ,
          --
          user_clk                => s_user_clk            ,
          sync_clk                => s_sync_clk            ,
          tx_out_clk              => tx_out_clk            ,
          gt0_qplllock_in         => gt0_qplllock_i        ,
          gt0_qpllrefclklost_in   => gt0_qpllrefclklost_i  ,
          gt_qpllclk_quad1_in     => gt_qpllclk_quad1_i    ,
          gt_qpllrefclk_quad1_in  => gt_qpllrefclk_quad1_i ,
          gt0_qpllreset_out       => gt0_qpllreset_i       ,
          pll_not_locked          => not s_tx_lock
        );
    end generate gen_x1_32b;

    -- Aurora IP - Framing - x1 - 16b - CRC
    gen_x1_16b_crc : if nb_lanes = 1 and lane_width = 16 and crc_enable = '1' generate

    end generate gen_x1_16b_crc;

    -- Aurora IP - Framing - x1 - 32b - CRC
    gen_x1_32b_crc : if nb_lanes = 1 and lane_width = 32 and crc_enable = '1' generate

    end generate gen_x1_32b_crc;

    -- Aurora IP - Framing - x2 - 16b
    gen_x2_16b : if nb_lanes = 2 and lane_width = 16 and crc_enable = '0' generate
      signal lane_xcvr_rx_p : STD_LOGIC_VECTOR(0 to 1);
      signal lane_xcvr_rx_n : STD_LOGIC_VECTOR(0 to 1);
      signal lane_xcvr_tx_p : STD_LOGIC_VECTOR(0 to 1);
      signal lane_xcvr_tx_n : STD_LOGIC_VECTOR(0 to 1);

    Begin

      i_aurora_8b10b_x2_16b : aurora_8b10b_x2_16b
        port map   (
          S_AXI_TX_TDATA          => s_pdu_tx_tdata ,
          S_AXI_TX_TKEEP          => s_pdu_tx_tkeep ,
          S_AXI_TX_TVALID         => s_pdu_tx_tvalid,
          S_AXI_TX_TLAST          => s_pdu_tx_tlast ,
          S_AXI_TX_TREADY         => s_pdu_tx_tready,
          M_AXI_RX_TDATA          => s_pdu_rx_tdata ,
          M_AXI_RX_TKEEP          => s_pdu_rx_tkeep ,
          M_AXI_RX_TVALID         => s_pdu_rx_tvalid,
          M_AXI_RX_TLAST          => s_pdu_rx_tlast ,
          S_AXI_NFC_TX_TREADY     => s_nfc_tx_tready,
          S_AXI_NFC_TX_TVALID     => s_nfc_tx_tvalid,
          S_AXI_NFC_TX_TDATA      => s_nfc_tx_tdata ,
          M_AXI_NFC_RX_TVALID     => s_nfc_rx_tvalid,
          M_AXI_NFC_RX_TDATA      => s_nfc_rx_tdata ,
          S_AXI_UFC_TX_TREADY     => s_ufc_tx_tready,
          S_AXI_UFC_TX_TVALID     => s_ufc_tx_tvalid,
          S_AXI_UFC_TX_TDATA      => s_ufc_tx_tdata ,
          M_AXI_UFC_RX_TVALID     => s_ufc_rx_tvalid,
          M_AXI_UFC_RX_TDATA      => s_ufc_rx_tdata ,
          M_AXI_UFC_RX_TKEEP      => s_ufc_rx_tkeep ,
          M_AXI_UFC_RX_TLAST      => s_ufc_rx_tlast ,
          -- XCVR Lanes
          RXP                     => lane_xcvr_rx_p ,
          RXN                     => lane_xcvr_rx_n ,
          TXP                     => lane_xcvr_tx_p ,
          TXN                     => lane_xcvr_tx_n ,
          -- GT Reference Clock
          gt_refclk1             => gt_clk          ,
          -- Error Detection Interface
          HARD_ERR                => s_hard_err     ,
          SOFT_ERR                => s_soft_err     ,
          FRAME_ERR               => s_frame_err    ,
          CHANNEL_UP              => s_channel_up   ,
          LANE_UP                 => s_lane_up      ,
          --
          reset                   => s_rst          ,
          power_down              => '0'            ,
          loopback                => (others=>'0')  , -- no Loopback of Transceiver
          gt_reset                => s_rst          ,
          tx_lock                 => s_tx_lock      ,
          sys_reset_out           => open           ,
          init_clk_in             => init_clk       ,
          tx_resetdone_out        => open           ,
          rx_resetdone_out        => open           ,
          link_reset_out          => open           ,
          drpclk_in               => '0'            ,
          drpaddr_in              => (others=>'0')  ,
          drpen_in                => '0'            ,
          drpdi_in                => (others=>'0')  ,
          drprdy_out              => open           ,
          drpdo_out               => open           ,
          drpwe_in                => '0'            ,
          drpaddr_in_lane1        => (others => '0'),
          drpen_in_lane1          => '0'            ,
          drpdi_in_lane1          => (others => '0'),
          drprdy_out_lane1        => open           ,
          drpdo_out_lane1         => open           ,
          drpwe_in_lane1          => '0'            ,
          --
          tx_out_clk              => tx_out_clk            ,
          user_clk                => s_user_clk            ,
          sync_clk                => s_sync_clk            ,
          gt0_qplllock_in         => gt0_qplllock_i        ,
          gt0_qpllrefclklost_in   => gt0_qpllrefclklost_i  ,
          gt_qpllclk_quad1_in     => gt_qpllclk_quad1_i    ,
          gt_qpllrefclk_quad1_in  => gt_qpllrefclk_quad1_i ,
          gt0_qpllreset_out       => gt0_qpllreset_i       ,
          pll_not_locked          => not s_tx_lock
        );
        --
        lane_xcvr_rx_p(1) <= xcvr_rx_p(1);
        lane_xcvr_rx_n(1) <= xcvr_rx_n(1);
        lane_xcvr_rx_p(0) <= xcvr_rx_p(0);
        lane_xcvr_rx_n(0) <= xcvr_rx_n(0);
        --
        xcvr_tx_p(1) <= lane_xcvr_tx_p(1);
        xcvr_tx_n(1) <= lane_xcvr_tx_n(1);
        xcvr_tx_p(0) <= lane_xcvr_tx_p(0);
        xcvr_tx_n(0) <= lane_xcvr_tx_n(0);
      end generate gen_x2_16b;

    -- Aurora IP - Framing - x4 - 16b
    gen_x4_16b : if nb_lanes = 4 and lane_width = 16 and crc_enable = '0' generate

    end generate gen_x4_16b;

    -- Aurora IP - Framing - x4 - 32b
    gen_x4_32b : if nb_lanes = 4 and lane_width = 32 and crc_enable = '0' generate

    end generate gen_x4_32b;

    -- -------------------------------------------
    -- -------------------------------------------
    --  PDU GEN
    -- -------------------------------------------
    -- -------------------------------------------

    -- -------------------------------------------
    --  PDU generator
    -- -------------------------------------------
    i_pdu_gen : entity work.pdu_gen
      generic map (
        nb_16b_lanes   => c_nb_16b_lanes, -- Number of 16bits lanes.
        counter_width  => frame_cnt_bits, -- Number of bits for the frame counter
        streaming_mode => streaming_mode
      )
      port map (
        rst_n            => s_channel_up        ,
        clk              => s_user_clk          ,
        pdu_rdy          => s_pdu_tx_rdy        ,
        pdu_sop          => s_pdu_tx_sop        ,
        pdu_vld          => s_pdu_tx_vld        ,
        pdu_d            => s_pdu_tx_d          ,
        pdu_bena         => s_pdu_tx_bena       ,
        pdu_eop          => s_pdu_tx_eop        ,
        fixed_size_on    => pdu_fixed_size_on   ,
        fixed_size_max   => pdu_fixed_size_max  ,
        fixed_size_min   => pdu_fixed_size_min  ,
        fixed_size_incr  => pdu_fixed_size_incr ,
        fixed_size_ifgap => pdu_fixed_size_ifgap,
        fixed_cut_size   => pdu_fixed_cut_size  ,
        fixed_cut_gap    => pdu_fixed_cut_gap   ,
        underflow_mode   => pdu_underflow       , -- '1' to generate not valid cycles
        endless_mode     => pdu_endless_mode    ,
        enable           => pdu_tx_ena          ,
        frame_cnt        => s_pdu_tx_frame_cnt  ,
        frame_size       => s_pdu_tx_frame_size ,
        reset_cnt        => reset_cnt
      );


    -- -------------------------------------------
    --  AVL ST to AXI interface
    -- -------------------------------------------
    i_pdu_if_to_axi : entity work.if_to_axi
      generic map (
        nb_lanes   => nb_lanes,
        lane_width => lane_width
      )
      port map (
        rst_n    => s_channel_up   ,
        clk      => s_user_clk     ,
        din_rdy  => s_pdu_tx_rdy   ,
        din_sop  => s_pdu_tx_sop   ,
        din_vld  => s_pdu_tx_vld   ,
        din_d    => s_pdu_tx_d     ,
        din_bena => s_pdu_tx_bena  ,
        din_eop  => s_pdu_tx_eop   ,
        axi_rdy  => s_pdu_tx_tready,
        axi_vld  => s_pdu_tx_tvalid,
        axi_d    => s_pdu_tx_tdata2,
        axi_keep => s_pdu_tx_tkeep ,
        axi_last => s_pdu_tx_tlast
      );

    -- PDU Tx Data : beware : UFC TX Data is also sent on PDU TX Data (multiplexing here)
    s_pdu_tx_tdata <= s_pdu_tx_tdata2(s_pdu_tx_tdata'range) when s_pdu_tx_tready = '1' else s_ufc_tx_data_f;


    -- -------------------------------------------
    --  PDU RX : AXI to generic interface
    -- -------------------------------------------
    i_pdu_rx_if_from_axi : entity work.if_from_axi
      generic map (
        nb_lanes   => nb_lanes  ,
        lane_width => lane_width
      )
      port map (
        rst_n     => s_channel_up   ,
        clk       => s_user_clk     ,
        axi_vld   => s_pdu_rx_tvalid,
        axi_d     => s_pdu_rx_tdata ,
        axi_keep  => s_pdu_rx_tkeep ,
        axi_last  => s_pdu_rx_tlast ,
        axi_rdy   => open           ,
        dout_d    => s_pdu_rx_d     ,
        dout_vld  => s_pdu_rx_vld   ,
        dout_bena => s_pdu_rx_bena  ,
        dout_sop  => s_pdu_rx_sop   ,
        dout_eop  => s_pdu_rx_eop   ,
        dout_rdy  => std_logic'('1')
      );

    -- -------------------------------------------
    --  PDU RX checker
    -- -------------------------------------------
    i_pdu_chk : entity work.pdu_chk
      generic map (
        nb_16b_lanes  => c_nb_16b_lanes, -- Number of 16bits lanes.
        counter_width => frame_cnt_bits  -- Number of bits for the frame counter
      )
      port map (
        rst_n     => s_channel_up      ,
        clk       => s_user_clk        ,
        pdu_sop   => s_pdu_rx_sop      ,
        pdu_vld   => s_pdu_rx_vld      ,
        pdu_d     => s_pdu_rx_d        ,
        pdu_bena  => s_pdu_rx_bena     ,
        pdu_eop   => s_pdu_rx_eop      ,
        pdu_err   => std_logic'('0')   ,
        frame_cnt => s_pdu_rx_frame_cnt,
        error_cnt => s_pdu_rx_error_cnt,
        reset_cnt => reset_cnt
      );
    pdu_rx_frame_cnt <= s_pdu_rx_frame_cnt;
    pdu_rx_error_cnt <= s_pdu_rx_error_cnt;

    -- synthesis translate_off
    -- PDU Data Received checking
    process
    begin
      wait until s_channel_up = '1' and simplex_rx = '0' and pdu_tx_ena = '1';
      wait until rising_edge(s_user_clk) and s_pdu_rx_vld = '1' for 30 us;
      if s_pdu_rx_vld /= '1' then
        report "No PDU RX Data Valid received 30us after link was up ... Probably a problem !" severity failure;
      end if;
    end process;
    -- synthesis translate_on


    -- -------------------------------------------
    -- -------------------------------------------
    --  UFC
    -- -------------------------------------------
    -- -------------------------------------------

    -- -------------------------------------------
    --  UFC TX generator
    -- -------------------------------------------
    i_ufc_gen : entity work.ufc_gen
      generic map (
        nb_16b_lanes  => c_nb_16b_lanes, -- Number of 16bits lanes
        counter_width => frame_cnt_bits, -- Number of bits for the frame counter
        auto_gen      => true          , -- If true, UFC are requested automatically, if false ufc_start must be asserted by the user.
        auto_gen_bits => 6             , -- Number of bits for the auto gen free run counter (UFC is sent when counter=0 and LFSR=1);
        ufc_direct    => '1'             -- UFC Direct if '1', Streaming if '0'
      )
      port map (
        rst_n     => s_channel_up      , -- in  std_logic       -- asynchronous active low reset (txclk domain)
        clk       => s_user_clk        , -- in  std_logic       -- clock for the TX path
        ufc_rdy   => s_ufc_tx_tready   , -- in  std_logic
        ufc_sop   => open              , -- out std_logic
        ufc_vld   => s_ufc_tx_tvalid   , -- out std_logic
        ufc_d     => s_ufc_tx_data_f   , -- out std_logic_vector(16*c_nb_16b_lanes-1 downto 0)
        ufc_bena  => open              , -- out std_logic_vector( 2*c_nb_16b_lanes-1 downto 0)
        ufc_eop   => open              , -- out std_logic
        ufc_nb    => s_ufc_tx_tdata    , -- out std_logic_vector(2 downto 0)    -- 0=2 bytes, 1=4 bytes, 2=6 bytes ...
        enable    =>   ufc_tx_ena      , -- in  std_logic
        ufc_start => std_logic'('0')   , -- in  std_logic                                       -- one cycle pulse to send an UFC.
        reset_cnt => reset_cnt         , -- in  std_logic
        frame_cnt => s_ufc_tx_frame_cnt  -- out std_logic_vector(COUNTER_WIDTH-1 downto 0)      -- number of frames sent.
      );

    -- -------------------------------------------
    --  UFC RX : AXI to generic interface
    -- -------------------------------------------
    i_ufc_rx_if_from_axi : entity work.if_from_axi
      generic map (
        nb_lanes   => nb_lanes  ,
        lane_width => lane_width
      )
      port map (
        rst_n     => s_channel_up   ,
        clk       => s_user_clk     ,
        axi_vld   => s_ufc_rx_tvalid,
        axi_d     => s_ufc_rx_tdata ,
        axi_keep  => s_ufc_rx_tkeep ,
        axi_last  => s_ufc_rx_tlast ,
        axi_rdy   => open           ,
        dout_d    => s_ufc_rx_d     ,
        dout_vld  => s_ufc_rx_vld   ,
        dout_bena => s_ufc_rx_bena  ,
        dout_sop  => s_ufc_rx_sop   ,
        dout_eop  => s_ufc_rx_eop   ,
        dout_rdy  => std_logic'('1')
      );

    -- -------------------------------------------
    --  UFC RX checker
    -- -------------------------------------------
    i_ufc_chk : entity work.ufc_chk
      generic map (
        nb_16b_lanes  => c_nb_16b_lanes,
        counter_width => frame_cnt_bits
      )
      port map (
        rst_n     => s_channel_up      ,
        clk       => s_user_clk        ,
        ufc_sop   => s_ufc_rx_sop      ,
        ufc_vld   => s_ufc_rx_vld      ,
        ufc_d     => s_ufc_rx_d        ,
        ufc_bena  => s_ufc_rx_bena     ,
        ufc_eop   => s_ufc_rx_eop      ,
        ufc_err   => std_logic'('0')   ,
        frame_cnt => s_ufc_rx_frame_cnt,
        reset_cnt => reset_cnt         ,
        error_cnt => s_ufc_rx_error_cnt,
        frame_ok  => s_ufc_rx_frame_ok ,
        frame_err => s_ufc_rx_frame_err
      );


    -- -------------------------------------------
    -- -------------------------------------------
    --  NFC
    -- -------------------------------------------
    -- -------------------------------------------

    -- -------------------------------------------
    --  NFC TX generator
    -- -------------------------------------------
    i_nfc_gen : entity work.nfc_gen
      generic map (
        counter_width => frame_cnt_bits, -- Number of bits for the frame counter
        auto_gen      => true          , -- If true, NFC are requested automatically, if false nfc_start must be asserted by the user.
        auto_gen_bits => 8               -- Number of bits for the auto gen free run counter (NFC is sent when counter=0 and LFSR=1);
      )
      port map (
        rst_n        => s_channel_up     ,
        clk          => s_user_clk       ,
        enable       =>   nfc_tx_ena     ,
        nfc_start    => std_logic'('0')  , -- 0's since auto_gen is true
        nfc_start_nb => (others => '0')  , -- 0's since auto_gen is true
        nfc_rdy      => s_nfc_tx_tready  ,
        nfc_vld      => s_nfc_tx_tvalid  ,
        nfc_d        => s_nfc_tx_tdata   ,
        wait_cnt     => s_nfc_tx_wait_cnt,
        reset_cnt    => reset_cnt
      );

    -- -------------------------------------------
    --  NFC RX checker
    -- -------------------------------------------
    i_nfc_chk : entity work.nfc_chk
      generic map (
        counter_width => frame_cnt_bits
      )
      port map (
        rst_n     => s_channel_up     ,
        clk       => s_user_clk       ,
        nfc_vld   => s_nfc_rx_tvalid  ,
        nfc_d     => s_nfc_rx_tdata   ,
        wait_cnt  => s_nfc_rx_wait_cnt,
        reset_cnt => reset_cnt
      );

  end generate gen_framing;



  -- ----------------------------------------------------
  --  Xilinx Aurora IP Instance, Streaming Mode
  -- ----------------------------------------------------
  gen_streaming : if streaming_mode = '1' generate

    -- Aurora IP - Streaming - x1 - 16b
    gen_x1_16b : if nb_lanes = 1 and lane_width = 16 generate

    end generate gen_x1_16b;

    -- Aurora IP - Streaming - x1 - 32b
    gen_x1_32b : if nb_lanes = 1 and lane_width = 32 generate

    end generate gen_x1_32b;

    -- Aurora IP - Streaming - x4 - 16b
    gen_x4_16b : if nb_lanes = 4 and lane_width = 16 generate

    end generate gen_x4_16b;

    -- Aurora IP - Streaming - x4 - 32b
    gen_x4_32b : if nb_lanes = 4 and lane_width = 32 generate

    end generate gen_x4_32b;


  end generate gen_streaming;


  user_clk <= s_user_clk;
  gt_clk_o <= gt_clk;

  -- ---------------------------
  --  Output Status
  -- ---------------------------
  status(15 downto 0) <= (others => '0');

  pdu_tx_frame_cnt  <= s_pdu_tx_frame_cnt ;
  pdu_tx_frame_size <= s_pdu_tx_frame_size;
  pdu_rx_frame_cnt  <= s_pdu_rx_frame_cnt;
  pdu_rx_error_cnt  <= s_pdu_rx_error_cnt;
  ufc_rx_frame_cnt  <= s_ufc_rx_frame_cnt;
  ufc_rx_error_cnt  <= s_ufc_rx_error_cnt;
  nfc_rx_wait_cnt   <= s_nfc_rx_wait_cnt;

--------------------------------------
end architecture rtl;
--------------------------------------

