///////////////////////////////////////////////////////////////////////////////
// (c) Copyright 2023 Advanced Micro Devices, Inc. All rights reserved.
//
// This file contains confidential and proprietary information
// of Advanced Micro Devices, Inc. and is protected under U.S. and
// international copyright and other intellectual property
// laws.
//
// DISCLAIMER
// This disclaimer is not a license and does not grant any
// rights to the materials distributed herewith. Except as
// otherwise provided in a valid license issued to you by
// AMD, and to the maximum extent permitted by applicable
// law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
// WITH ALL FAULTS, AND AMD HEREBY DISCLAIMS ALL WARRANTIES
// AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
// BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
// INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
// (2) AMD shall not be liable (whether in contract or tort,
// including negligence, or under any other theory of
// liability) for any loss or damage of any kind or nature
// related to, arising under or in connection with these
// materials, including for any direct, or any indirect,
// special, incidental, or consequential loss or damage
// (including loss of data, profits, goodwill, or any type of
// loss or damage suffered as a result of any action brought
// by a third party) even if such damage or loss was
// reasonably foreseeable or AMD had been advised of the
// possibility of the same.
//
// CRITICAL APPLICATIONS
// AMD products are not designed or intended to be fail-
// safe, or for use in any application requiring fail-safe
// performance, such as life-support or safety devices or
// systems, Class III medical devices, nuclear facilities,
// applications related to the deployment of airbags, or any
// other applications that could lead to death, personal
// injury, or severe property or environmental damage
// (individually and collectively, "Critical
// Applications"). Customer assumes the sole risk and
// liability of any use of AMD products in Critical
// Applications, subject only to applicable laws and
// regulations governing limitations on product liability.
//
// THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
// PART OF THIS FILE AT ALL TIMES.
//
//
///////////////////////////////////////////////////////////////////////////////
//
//  TX_LL_CONTROL
//
//
//  Description: This module provides the transmitter state machine
//               control logic to connect the LocalLink interface to
//               the Aurora Channel.
//
//               This module supports 1 2-byte lane designs
//
//               This module supports Immediate Mode Native Flow Control.
//
//               This module supports User Flow Control.
//

`timescale 1 ns / 1 ps

module  aurora_8b10b_x1_16b_TX_LL_CONTROL
(
    // LocalLink PDU Interface
    TX_SRC_RDY_N,
    TX_SOF_N,
    TX_EOF_N,
    TX_REM,

    TX_DST_RDY_N,

    // NFC Interface
    NFC_REQ_N,
    NFC_NB,

    NFC_ACK_N,
    // UFC Interface
    UFC_TX_REQ_N,
    UFC_TX_MS,

    UFC_TX_ACK_N,

    // Clock Compensation Interface
    WARN_CC,
    DO_CC,


    // Global Logic Interface
    CHANNEL_UP,


    // TX_LL Control Module Interface
    HALT_C,
    UFC_MESSAGE,


    // Aurora Lane Interface
    GEN_SCP,
    GEN_ECP,
    GEN_SNF,
    GEN_SUF,
    FC_NB,
    GEN_CC,

    // RX_LL Interface
    TX_WAIT,

    DECREMENT_NFC,

    // System Interface
    USER_CLK

);

`define DLY #1


//***********************************Port Declarations*******************************


    // LocalLink PDU Interface
input              TX_SRC_RDY_N;
input              TX_SOF_N;
input              TX_EOF_N;
input              TX_REM;

output             TX_DST_RDY_N;

    // NFC Interface
input              NFC_REQ_N;
input   [0:3]      NFC_NB;

output             NFC_ACK_N;


    // UFC Interface
input              UFC_TX_REQ_N;
input   [0:3]      UFC_TX_MS;

output             UFC_TX_ACK_N;

    // Clock Compensation Interface
input              WARN_CC;
input              DO_CC;


    // Global Logic Interface
input              CHANNEL_UP;

    // TX_LL Control Module Interface
output             HALT_C;
output             UFC_MESSAGE;

    // Aurora Lane Interface
output             GEN_SCP;
output             GEN_ECP;
output             GEN_SNF;
output             GEN_SUF;
output  [0:3]      FC_NB;
output             GEN_CC;

    // RX_LL Interface
input              TX_WAIT;

output             DECREMENT_NFC;

    // System Interface
input              USER_CLK;



//**************************External Register Declarations****************************

reg                TX_DST_RDY_N;
reg                GEN_SCP;
reg                GEN_ECP;
reg                GEN_SNF;
reg                GEN_SUF;
reg     [0:3]      FC_NB;


//**************************Internal Register Declarations****************************

reg                do_cc_r;
reg                do_nfc_r;
reg                ufc_idle_r;
reg                ufc_header_r;
reg                ufc_message1_r;
reg                ufc_message2_r;
reg                ufc_message3_r;
reg                ufc_message4_r;
reg                ufc_message5_r;
reg                ufc_message6_r;
reg                ufc_message7_r;
reg                ufc_message8_r;

reg     [0:2]      ufc_message_count_r;

reg                suf_delay_1_r;
reg                suf_delay_2_r;

reg     [0:3]      delay_ms_1_r;
reg     [0:3]      delay_ms_2_r;

reg                previous_cycle_ufc_message_r;
reg                create_gap_for_scp_r;

reg                idle_r;
reg                sof_r;
reg                sof_data_eof_1_r;
reg                sof_data_eof_2_r;
reg                sof_data_eof_3_r;
reg                data_r;
reg                data_eof_1_r;
reg                data_eof_2_r;
reg                data_eof_3_r;





//*********************************Wire Declarations**********************************
wire               nfc_ok_c;
wire               next_ufc_idle_c;
wire               next_ufc_header_c;
wire               next_ufc_message1_c;
wire               next_ufc_message2_c;
wire               next_ufc_message3_c;
wire               next_ufc_message4_c;
wire               next_ufc_message5_c;
wire               next_ufc_message6_c;
wire               next_ufc_message7_c;
wire               next_ufc_message8_c;
wire               ufc_ok_c;
wire               create_gap_for_scp_c;

wire               next_idle_c;
wire               next_sof_c;
wire               next_sof_data_eof_1_c;
wire               next_sof_data_eof_2_c;
wire               next_sof_data_eof_3_c;
wire               next_data_c;
wire               next_data_eof_1_c;
wire               next_data_eof_2_c;
wire               next_data_eof_3_c;



wire    [0:3]      fc_nb_c;
wire               tx_dst_rdy_n_c;
wire               do_sof_c;
wire               do_eof_c;
wire               channel_full_c;
wire               pdu_ok_c;
wire               reset_i;


//*********************************Main Body of Code**********************************

    assign reset_i   = ~CHANNEL_UP;

    //___________________________Clock Compensation________________________________


    // Register the DO_CC and WARN_CC signals for internal use.  Note that the raw DO_CC
    // signal is used for some logic so the DO_CC signal should be driven directly
    // from a register whenever possible.

    always @(posedge USER_CLK)
                       do_cc_r <=  `DLY    DO_CC;

    //_____________________________NFC State Machine__________________________________

    // The NFC state machine has 2 states: waiting for an NFC request, and
    // sending an NFC message.  It can take over the channel at any time
    // except when there is a UFC message or a CC sequence in progress.

    always @(posedge USER_CLK)
        if(!CHANNEL_UP)     do_nfc_r    <=  `DLY    1'b0;
        else if(!do_nfc_r)  do_nfc_r    <=  `DLY    !NFC_REQ_N & nfc_ok_c;
        else                do_nfc_r    <=  `DLY    1'b0;



    // You can only send an NFC message when there is no CC operation or UFC
    // message in progress.  We also prohibit NFC messages just before CC to
    // prevent collisions on the first cycle.
    assign  nfc_ok_c    =   !DO_CC &

                            ufc_idle_r &
                            !previous_cycle_ufc_message_r &
                            !WARN_CC;


    assign  NFC_ACK_N   =   !do_nfc_r;


    //_____________________________UFC State Machine__________________________________


    // The UFC state machine has 10 states: waiting for a UFC request, sending
    // a UFC header, and 8 states for sending up to 8 words of a UFC message.
    // It can take over the channel at any time except when there is an NFC
    // message or a CC sequence being sent.

    always @(posedge USER_CLK)
        if(!CHANNEL_UP)
            begin
                ufc_idle_r      <=  `DLY    1'b1;
                ufc_header_r    <=  `DLY    1'b0;
                ufc_message1_r  <=  `DLY    1'b0;
                ufc_message2_r  <=  `DLY    1'b0;
                ufc_message3_r  <=  `DLY    1'b0;
                ufc_message4_r  <=  `DLY    1'b0;
                ufc_message5_r  <=  `DLY    1'b0;
                ufc_message6_r  <=  `DLY    1'b0;
                ufc_message7_r  <=  `DLY    1'b0;
                ufc_message8_r  <=  `DLY    1'b0;
            end
        else
            begin
                ufc_idle_r      <=  `DLY    next_ufc_idle_c;
                ufc_header_r    <=  `DLY    next_ufc_header_c;
                ufc_message1_r  <=  `DLY    next_ufc_message1_c;
                ufc_message2_r  <=  `DLY    next_ufc_message2_c;
                ufc_message3_r  <=  `DLY    next_ufc_message3_c;
                ufc_message4_r  <=  `DLY    next_ufc_message4_c;
                ufc_message5_r  <=  `DLY    next_ufc_message5_c;
                ufc_message6_r  <=  `DLY    next_ufc_message6_c;
                ufc_message7_r  <=  `DLY    next_ufc_message7_c;
                ufc_message8_r  <=  `DLY    next_ufc_message8_c;
            end


    // Capture the message count so it can be used to determine the appropriate
    // next state.
    always @(posedge USER_CLK)
        if(next_ufc_header_c)  ufc_message_count_r <=   `DLY    UFC_TX_MS[0:2];




    assign  next_ufc_idle_c     =   ((UFC_TX_REQ_N | !ufc_ok_c)&
                                     ((ufc_idle_r)|
                                      (ufc_message1_r & (ufc_message_count_r==3'b000))|
                                      (ufc_message2_r & (ufc_message_count_r==3'b001))|
                                      (ufc_message3_r & (ufc_message_count_r==3'b010))|
                                      (ufc_message4_r & (ufc_message_count_r==3'b011))|
                                      (ufc_message5_r & (ufc_message_count_r==3'b100))|
                                      (ufc_message6_r & (ufc_message_count_r==3'b101))|
                                      (ufc_message7_r & (ufc_message_count_r==3'b110))|
                                      (ufc_message8_r & (ufc_message_count_r==3'b111))));


    assign  next_ufc_header_c   =   ((!UFC_TX_REQ_N & ufc_ok_c)&
                                     ((ufc_idle_r)|
                                      (ufc_message1_r & (ufc_message_count_r==3'b000))|
                                      (ufc_message2_r & (ufc_message_count_r==3'b001))|
                                      (ufc_message3_r & (ufc_message_count_r==3'b010))|
                                      (ufc_message4_r & (ufc_message_count_r==3'b011))|
                                      (ufc_message5_r & (ufc_message_count_r==3'b100))|
                                      (ufc_message6_r & (ufc_message_count_r==3'b101))|
                                      (ufc_message7_r & (ufc_message_count_r==3'b110))|
                                      (ufc_message8_r & (ufc_message_count_r==3'b111)) ));



    assign  next_ufc_message1_c =   ufc_header_r;

    assign  next_ufc_message2_c =   ufc_message1_r  & (ufc_message_count_r!=3'b000);

    assign  next_ufc_message3_c =   ufc_message2_r  & (ufc_message_count_r!=3'b001);

    assign  next_ufc_message4_c =   ufc_message3_r  & (ufc_message_count_r!=3'b010);

    assign  next_ufc_message5_c =   ufc_message4_r  & (ufc_message_count_r!=3'b011);

    assign  next_ufc_message6_c =   ufc_message5_r  & (ufc_message_count_r!=3'b100);

    assign  next_ufc_message7_c =   ufc_message6_r  & (ufc_message_count_r!=3'b101);

    assign  next_ufc_message8_c =   ufc_message7_r  & (ufc_message_count_r!=3'b110);


    assign  UFC_MESSAGE         = !ufc_idle_r & !ufc_header_r;






    assign  ufc_ok_c    =   !DO_CC & !WARN_CC & (NFC_REQ_N | do_nfc_r);

    assign  UFC_TX_ACK_N    =   !ufc_header_r;


    // Delay UFC_TX_MS so it arrives at the lanes at the same time as the
    // UFC header.
    always @(posedge USER_CLK)
    begin
        delay_ms_1_r    <=  `DLY    UFC_TX_MS;
        delay_ms_2_r    <=  `DLY    delay_ms_1_r;
    end


    always @(posedge USER_CLK)
    previous_cycle_ufc_message_r    <=  `DLY    !ufc_idle_r & !ufc_header_r;


    //_____________________________PDU State Machine__________________________________

    // The PDU state machine handles the encapsulation and transmission of user
    // PDUs.  It can use the channel when there is no CC, NFC message, UFC header,
    // UFC message or remote NFC request.





    // State Registers
    always @(posedge USER_CLK)
        if(!CHANNEL_UP)
        begin
            idle_r              <=  `DLY    1'b1;
            sof_r               <=  `DLY    1'b0;
            sof_data_eof_1_r    <=  `DLY    1'b0;
            sof_data_eof_2_r    <=  `DLY    1'b0;
            sof_data_eof_3_r    <=  `DLY    1'b0;
            data_r              <=  `DLY    1'b0;
            data_eof_1_r        <=  `DLY    1'b0;
            data_eof_2_r        <=  `DLY    1'b0;
            data_eof_3_r        <=  `DLY    1'b0;
        end
        else if(pdu_ok_c)
        begin
            idle_r              <=  `DLY    next_idle_c;
            sof_r               <=  `DLY    next_sof_c;
            sof_data_eof_1_r    <=  `DLY    next_sof_data_eof_1_c;
            sof_data_eof_2_r    <=  `DLY    next_sof_data_eof_2_c;
            sof_data_eof_3_r    <=  `DLY    next_sof_data_eof_3_c;
            data_r              <=  `DLY    next_data_c;
            data_eof_1_r        <=  `DLY    next_data_eof_1_c;
            data_eof_2_r        <=  `DLY    next_data_eof_2_c;
            data_eof_3_r        <=  `DLY    next_data_eof_3_c;
        end




    // Next State Logic
    assign  next_idle_c             =   (idle_r & !do_sof_c) |
                                        (sof_data_eof_3_r & !do_sof_c) |
                                        (data_eof_3_r & !do_sof_c );



    assign  next_sof_c              =   (idle_r & do_sof_c & !do_eof_c) |
                                        (sof_data_eof_3_r & do_sof_c & !do_eof_c) |
                                        (data_eof_3_r & do_sof_c & !do_eof_c);


    assign  next_data_c             =   (sof_r & !do_eof_c) |
                                        (data_r & !do_eof_c);


    assign  next_data_eof_1_c       =   (sof_r & do_eof_c)|
                                        (sof_data_eof_1_r & ufc_header_r)|
                                        (data_eof_1_r & ufc_header_r)|
                                        (data_r & do_eof_c);


    assign  next_data_eof_2_c       =   (data_eof_1_r & !ufc_header_r)|
                                        (data_eof_2_r & (previous_cycle_ufc_message_r | suf_delay_1_r));


    assign  next_data_eof_3_c       =   data_eof_2_r & !(previous_cycle_ufc_message_r | suf_delay_1_r);


    assign  next_sof_data_eof_1_c   =   (idle_r & do_sof_c & do_eof_c)|
                                        (sof_data_eof_3_r & do_sof_c & do_eof_c)|
                                        (data_eof_3_r & do_sof_c & do_eof_c);


    assign  next_sof_data_eof_2_c   =   (sof_data_eof_1_r & !ufc_header_r)|
                                        sof_data_eof_2_r & (previous_cycle_ufc_message_r | suf_delay_1_r);

    assign  next_sof_data_eof_3_c   =   sof_data_eof_2_r & !(previous_cycle_ufc_message_r | suf_delay_1_r);


    // Generate SCP characters whenever the PDU state machine is active in an SOF state.
    always @(posedge USER_CLK)
        if(!CHANNEL_UP) GEN_SCP <=  `DLY    1'b0;
        else            GEN_SCP <=  `DLY    ((sof_r | sof_data_eof_1_r) & pdu_ok_c);


    // Generate ECP characters whenever the PDU state machine is active in the final EOF state.
    always @(posedge USER_CLK)
        if(!CHANNEL_UP) GEN_ECP <=  `DLY    1'b0;
        else            GEN_ECP <=  `DLY    (data_eof_3_r | sof_data_eof_3_r) & pdu_ok_c;




    assign  tx_dst_rdy_n_c  =   (next_sof_data_eof_1_c & pdu_ok_c) |
                                sof_data_eof_1_r |
                                (next_data_eof_1_c & pdu_ok_c) |
                                !next_ufc_idle_c |
                                (!do_nfc_r & !NFC_REQ_N & nfc_ok_c) |
                                DO_CC  |
                                create_gap_for_scp_c |
                                TX_WAIT |
                                data_eof_1_r|
                                (data_eof_2_r && (!pdu_ok_c || previous_cycle_ufc_message_r)) |
                                (sof_data_eof_2_r && (!pdu_ok_c || previous_cycle_ufc_message_r));






    // SCP characters can only be added when the first lane position is open.  After UFC messages,
    // data is deliberately held off for one cycle to create this gap.  No gap is added if no
    // SCP character is needed.
    assign  create_gap_for_scp_c    =   !ufc_idle_r && !ufc_header_r &&
                                        (idle_r ||
                                        data_eof_3_r ||
                                        sof_data_eof_3_r);

    always @(posedge USER_CLK)
        create_gap_for_scp_r    <=  `DLY    create_gap_for_scp_c;











    // The flops for the GEN_CC signal are replicated for timing and instantiated to allow us
    // to set their value reliably on powerup.
    FDR gen_cc_flop_0_i
    (
        .D(do_cc_r),
        .C(USER_CLK),
        .R(1'b0),//.(reset_i),
        .Q(GEN_CC)
    );






    // GEN_SNF is asserted whenever the NFC state machine is not idle.
    always @(posedge USER_CLK)
        if(!CHANNEL_UP) GEN_SNF <=  `DLY    1'b0;
        else            GEN_SNF <=  `DLY    do_nfc_r;





    // The UFC header state triggers the generation of SUF characters in the lane.  The signal is
    // delayed to match up with the datapath delay so that SUF always appears on the cycle
    // before the first data byte.
    always @(posedge USER_CLK)
        if(!CHANNEL_UP) suf_delay_1_r   <=  `DLY    1'b0;
        else            suf_delay_1_r   <=  `DLY    ufc_header_r;



    always @(posedge USER_CLK)
        if(!CHANNEL_UP) suf_delay_2_r   <=  `DLY    1'b0;
        else            suf_delay_2_r   <=  `DLY    suf_delay_1_r;


    always @(posedge USER_CLK)
        if(!CHANNEL_UP) GEN_SUF <=  `DLY    1'b0;
        else            GEN_SUF <=  `DLY    suf_delay_2_r;




    // FC_NB carries flow control codes to the Lane Logic.
    always @(posedge USER_CLK)
        FC_NB   <=  `DLY    fc_nb_c;




    // Flow control codes come from the NFC_NB input unless the UFC state machine is actively
    // sending an SUF character.  When UFC is active, the code comes from the UFC_TX_MS input
    // delayed to match the UFC data delay.
    assign  fc_nb_c =   suf_delay_2_r?delay_ms_2_r:NFC_NB;



    // The TX_DST_RDY_N signal is registered.
    always @(posedge USER_CLK)
        if(!CHANNEL_UP)     TX_DST_RDY_N    <=  `DLY    1'b1;
        else                TX_DST_RDY_N    <=  `DLY    tx_dst_rdy_n_c;



    // Decrement the NFC pause required count whenever the state machine prevents new
    // PDU data from being sent except when the data is prevented by CC characters.
    assign DECREMENT_NFC = TX_DST_RDY_N && !do_cc_r;





    // Helper Logic
   
   
   
    // SOF requests are valid when TX_SRC_RDY_N, TX_DST_RDY_N and TX_SOF_N are all asserted
    assign  do_sof_c                =   !TX_SRC_RDY_N &
                                        !TX_DST_RDY_N &
                                        !TX_SOF_N;


    // EOF requests are valid when TX_SRC_RDY_N, TX_DST_RDY_N and TX_EOF_N are all asserted
    assign  do_eof_c                =   !TX_SRC_RDY_N &
                                        !TX_DST_RDY_N &
                                        !TX_EOF_N;
   
   
   
   
   
   




    // Freeze the PDU state machine when CCs or NFCs must be handled.  Note that the PDU state
    // machine does not freeze for UFCs - instead, logic is provided to allow the two datastreams
    // to cooperate.
    assign  pdu_ok_c                =   !do_cc_r &
                                        !do_nfc_r;

    // Halt the flow of data through the datastream when the PDU state machine is frozen or
    // when an SCP character has been delayed due to UFC collision.
    assign  HALT_C                  =   !pdu_ok_c;







    // The aurora channel is 'full' if there is more than enough data to fit into
    // a channel that is already carrying an SCP and an ECP character.
    assign  channel_full_c          =   1'b1;

endmodule
