----------------------------------------------------------------------------------------------------
-- Copyright (c) ReFLEX CES 1998-2012
--
-- Use of this source code through a simulator and/or a compiler tool
-- is illegal if not authorised through ReFLEX CES License agreement.
----------------------------------------------------------------------------------------------------
-- Project     : 
----------------------------------------------------------------------------------------------------
-- Top level   : top.vhd
-- File        : ufc_gen.vhd
-- Author      : flavenant@reflexces.com
-- Company     : ReFLEX CES
--               2, rue du gevaudan
--               91047 LISSES
--               FRANCE
--               http://www.reflexces.com
-- Plateforme  : Windows XP
-- Simulator   : Mentor Graphics ModelSim
-- Synthesis   : Quartus II
-- Target      : Stratix IV
-- Dependency  :
----------------------------------------------------------------------------------------------------
-- Description :
-- UFC generator
--  
----------------------------------------------------------------------------------------------------
-- Version      Date            Author              Description
-- 0.1          2012/01/16      FLA                 Creation
-- 0.2          2019/05/31      SL&BC               Removed std_logic_unsigned and reduced
--                                                  the range of the always-even s_data_byte_base
--                                                  counter
----------------------------------------------------------------------------------------------------
library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;
--use     ieee.std_logic_unsigned.all;

entity ufc_gen is
    generic (
          NB_16B_LANES   : integer := 5          -- Number of 16bits lanes.
        ; COUNTER_WIDTH  : integer := 32         -- Number of bits for the frame counter
        ; AUTO_GEN       : boolean := true       -- If true, UFC are requested automatically, if false ufc_start must be asserted by the user.
        ; AUTO_GEN_BITS  : integer := 6          -- Number of bits for the auto gen free run counter (UFC is sent when counter=0 and LFSR=1);
        ; UFC_DIRECT     : std_logic := '1'          -- UFC Direct (using tx_ufc_nb) is used if '1', otherwise UFC "Streaming"
    );
    port (
        -- Reset and clocks
          rst_n                 : in    std_logic
        ; clk                   : in    std_logic
        
        -- UFC
        ; ufc_rdy               : in    std_logic
        ; ufc_sop               : out   std_logic
        ; ufc_vld               : out   std_logic
        ; ufc_d                 : out   std_logic_vector(16*NB_16B_LANES-1 downto 0)
        ; ufc_bena              : out   std_logic_vector( 2*NB_16B_LANES-1 downto 0)
        ; ufc_eop               : out   std_logic
        ; ufc_nb                : out   std_logic_vector(2 downto 0)    -- 0=2 bytes, 1=4 bytes, 2=6 bytes ...
        
        -- Control / Status
        ; enable                : in    std_logic                                       
        ; ufc_start             : in    std_logic                                       -- one cycle pulse to send an UFC.
        ; frame_cnt             : out   std_logic_vector(COUNTER_WIDTH-1 downto 0)      -- number of frames sent.
        ; reset_cnt             : in    std_logic
    );
end entity ufc_gen;

architecture rtl of ufc_gen is
	----------------------------------------------------------------
	-- Type declarations
	----------------------------------------------------------------
    type TState is (FS_REQ, FS_ACK, FS_SEND);
    
	----------------------------------------------------------------
	-- Function declarations
	----------------------------------------------------------------
    -- Round to floor Log2
    function MyLocalLog2(nb : integer) return integer is
        variable v_rtn : integer := 0;
    begin
        if nb=0 then v_rtn := 0;
        else         while (nb>2**v_rtn) loop v_rtn := v_rtn + 1; end loop;
        end if;
        return v_rtn;
    end function MyLocalLog2;
        
    -- LFSR (Linear Feedback Shift Register) function for test
    -- /!\ use xnor to set 111...11 as the idle state instead of 000...00
    -- '1' bit in mask enables xnor with corresponding bit in vectIn
    function DoLFSR(vectIn : std_logic_vector;
                    mask   : std_logic_vector ) return std_logic_vector is
        variable v_rtn : std_logic_vector(vectIn'range);
        variable v_fb  : std_logic;
    begin
        v_fb              := vectIn(0);
        v_rtn(v_rtn'high) := v_fb;
    
        for i in vectIn'low to vectIn'high-1 loop
			if (mask(i+1)='1') then v_rtn(i) := vectIn(i+1) xnor v_fb;
			else                    v_rtn(i) := vectIn(i+1);
			end if;
        end loop;
        
        return v_rtn;
    end function DoLFSR;
    
	----------------------------------------------------------------
	-- Component declarations
	----------------------------------------------------------------
  component scfifo is
    generic (
      intended_device_family  : string;
      lpm_type                : string;
      lpm_numwords            : natural;
      lpm_width               : natural;
      lpm_widthu              : natural;
      lpm_showahead           : string;
      use_eab                 : string;
      overflow_checking       : string;
      underflow_checking      : string;
      add_ram_output_register : string
    );
    port (
      aclr  : in  std_logic;
      clock : in  std_logic;
      sclr  : in  std_logic;
      wrreq : in  std_logic;
      data  : in  std_logic_vector(lpm_width-1 downto 0);
      full  : out std_logic;
      usedw : out std_logic_vector(lpm_widthu-1 downto 0);
      empty : out std_logic ;
      rdreq : in  std_logic ;
      q     : out std_logic_vector(lpm_width-1 downto 0)
    );
  end component scfifo;
    
	----------------------------------------------------------------
	-- Constant declarations
	----------------------------------------------------------------
    constant C_LFSR_REQ                 : std_logic_vector(28 downto 0) := std_logic_vector(to_unsigned(16#4f3#, 29)); 
    constant C_LFSR_NB                  : std_logic_vector(28 downto 0) := std_logic_vector(to_unsigned(16#4f3#, 29)); 
    constant C_LFSR_TEMPO               : std_logic_vector(27 downto 0) := std_logic_vector(to_unsigned(16#4f3#, 28)); 

	----------------------------------------------------------------
	-- Signal declarations
	----------------------------------------------------------------
    signal s_state              : TState;
    signal s_do_req             : std_logic;
    signal s_ufc_vld            : std_logic;
    signal s_ufc_rdy            : std_logic;
    signal s_ufc_nb             : std_logic_vector(ufc_nb'range);
    signal s_ufc_d              : std_logic_vector(ufc_d'range);
    signal s_req_nb             : std_logic_vector(ufc_nb'range);
    signal s_beat_cnt           : unsigned(ufc_nb'high+1 downto 0);
    signal s_incr               : unsigned(ufc_nb'high+1 downto 0);
    -- 2019/05/31: reduce 8 bits counter to 7 bits (bit 0 was unused => Quartus Pro wrongfully reports a latch)
    signal s_data_byte_base     : unsigned(6 downto 0);
    signal s_tempo_cnt          : unsigned(AUTO_GEN_BITS-1 downto 0);
    signal s_tempo_cnt0         : std_logic;
    signal s_lfsr_req           : std_logic_vector(C_LFSR_REQ'range);
    signal s_lfsr_nb            : std_logic_vector(C_LFSR_NB'range);
    signal s_lfsr_tempo         : std_logic_vector(C_LFSR_TEMPO'range);
    signal s_frame_cnt          : unsigned(COUNTER_WIDTH-1 downto 0);
    
  -- UFC Fifo Signals (when in UFC Streaming mode : ufc_direct = '0')  
  constant c_fifo_depth : positive := 8; -- 2**fifo_depth words in Fifo : depends on ufc_d'length
  constant c_fifo_wsize : positive := ufc_bena'length+ufc_d'length+1+1;
  signal s_fifo_aclr    : std_logic;
  signal s_fifo_wrreq   : std_logic;
  signal s_fifo_wrdata  : std_logic_vector(c_fifo_wsize-1 downto 0);
  signal s_fifo_full    : std_logic;         
  signal s_fifo_empty   : std_logic;         
  signal s_fifo_rdreq   : std_logic;         
  signal s_fifo_rddata  : std_logic_vector(c_fifo_wsize-1 downto 0);
  type t_tx_fsm is ( s_idle, s_store );
  signal s_tx_fsm       : t_tx_fsm;
  signal s_tx_cnt       : unsigned(4 downto 0); -- to count up to 16
  signal s_tx_len       : unsigned(4 downto 0); -- to count up to 16 
  
  function calc_byte_enables ( length : positive; nb_words : positive; data_width : positive ) return std_logic_vector is
    variable v_rem  : integer;
    variable be_res : std_logic_vector(data_width/8-1 downto 0);
  begin
    be_res := (others => '1'); -- default : all asserted
    v_rem := nb_words * data_width/8 - length; -- Total Bytes
    if v_rem > 0 then
      for i in 0 to data_width/8-1 loop
        if i >= data_width/8-v_rem then
          be_res(i) := '0';
        end if;
      end loop;
    end if;
    return be_res;
  end function calc_byte_enables;
          
  function or_ed ( d : unsigned ) return unsigned is
    variable v_or  : std_logic;    
    variable v_res : unsigned(0 downto 0);    
  begin
    v_or := '0';
    for i in d'range loop
      v_or := v_or or d(i);
    end loop;
    v_res(0) := v_or; 
    return v_res;    
  end function or_ed;
  
  
begin

    --##############################################################
    -- Main process
    --##############################################################     
    p_main : process (clk, rst_n)
        --variable v_data_byte_base   : std_logic_vector(s_data_byte_base'range);
        variable v_do_req           : std_logic;
    begin
    if rst_n='0' then 
        s_lfsr_req          <= C_LFSR_REQ;
        s_lfsr_nb           <= C_LFSR_NB;
        s_lfsr_tempo        <= C_LFSR_TEMPO;
        s_state             <= FS_REQ;
        s_beat_cnt          <= (others=>'0');
        s_incr              <= (others=>'0');
        s_data_byte_base    <= (others=>'0');
        s_tempo_cnt         <= (others=>'0');
        s_tempo_cnt0        <= '0';
        s_do_req            <= '0';
        s_ufc_nb            <= (others=>'0');
        s_ufc_vld           <= '0';        
        s_ufc_d             <= (others=>'0');
        s_req_nb            <= (others=>'0');
        s_frame_cnt         <= (others=>'0');
        
    elsif rising_edge(clk) then
      
        ----------------------------------------------------------------------------------------------------
        -- Manage frame counter
        ----------------------------------------------------------------------------------------------------
        -- Count frames sent
           if reset_cnt='1'                                       then s_frame_cnt <= (others=>'0');
        elsif s_state=FS_SEND and s_beat_cnt(s_beat_cnt'high)='1' then s_frame_cnt <= s_frame_cnt + 1; end if;
    
        
        ----------------------------------------------------------------------------------------------------
        -- UFC requester
        ----------------------------------------------------------------------------------------------------
        -- Assert this variable when a request is to be sent
           if AUTO_GEN=true  and s_state=FS_REQ then v_do_req := s_lfsr_req(0) and s_tempo_cnt0 and enable;
        elsif AUTO_GEN=false and s_state=FS_REQ then v_do_req := ufc_start     and                  enable;
        else                                         v_do_req := '0'; end if;
        
        -- Free counter
        s_lfsr_tempo <= DoLFSR(s_lfsr_tempo, C_LFSR_TEMPO);
        if s_tempo_cnt=0 then s_tempo_cnt0 <= '1';
        else                  s_tempo_cnt0 <= '0'; end if;
        
        if s_tempo_cnt=0 then s_tempo_cnt <= (others=>'1'); s_tempo_cnt(2 downto 0) <= unsigned(s_lfsr_tempo(2 downto 0));
        else                  s_tempo_cnt <= s_tempo_cnt - 1; end if;
        
        -- LFSR for random requests
        if s_tempo_cnt0='1' then s_lfsr_req <= DoLFSR(s_lfsr_req, C_LFSR_REQ); end if;
        
        -- UFC request
           if v_do_req='1'    then s_do_req <= '1';
        elsif s_state=FS_SEND then s_do_req <= '0'; end if;
        
        -- UFC request NB
        s_lfsr_nb <= DoLFSR(s_lfsr_nb, C_LFSR_NB);
        if v_do_req='1' then s_req_nb <= s_lfsr_nb(s_req_nb'range); end if;
        
        --if v_do_req='1' then s_req_nb <= std_logic_vector(to_unsigned(NB_16B_LANES-1, s_req_nb'length)); end if;
        
        --if v_do_req='1' then
        --    if (unsigned(s_lfsr_nb(s_req_nb'range)) mod NB_16B_LANES)=NB_16B_LANES-1 and NB_16B_LANES>1 then s_req_nb <= s_lfsr_nb(s_req_nb'range) + 1;
        --    else                                                                                             s_req_nb <= s_lfsr_nb(s_req_nb'range); end if;
        --end if;
        
        ----------------------------------------------------------------------------------------------------
        -- Manage FSM
        ----------------------------------------------------------------------------------------------------
        -- FSM
        case s_state is
            when FS_REQ  => if s_do_req='1'                     then s_state <= FS_ACK; end if;
            when FS_ACK  => if s_ufc_rdy='1'                    then s_state <= FS_SEND; end if;
            when FS_SEND => if s_beat_cnt(s_beat_cnt'high)='1'  then s_state <= FS_REQ; end if;
        end case;
    
        -- Manage beat counter
           if NB_16B_LANES>7                   then s_beat_cnt <= (others=>'1'); -- to avoid overlaps of counter
        elsif s_state=FS_REQ and s_do_req='1'  then s_beat_cnt <= unsigned('0' & s_req_nb);
        elsif s_state=FS_SEND or s_ufc_rdy='1' then s_beat_cnt <= s_beat_cnt - NB_16B_LANES; end if;
        
        ----------------------------------------------------------------------------------------------------
        -- Manage data bytes
        ----------------------------------------------------------------------------------------------------
        -- Bytes increment
           if s_state=FS_REQ and s_do_req='1'                              then s_incr <= unsigned('0' & s_req_nb) + 1;
        elsif (s_state=FS_SEND or s_ufc_rdy='1') and s_incr>=NB_16B_LANES  then s_incr <= s_incr - NB_16B_LANES;
        elsif (s_state=FS_SEND or s_ufc_rdy='1') and s_incr<NB_16B_LANES   then s_incr <= (others=>'0'); end if;
        
        -- Data bytes
        --v_data_byte_base := s_data_byte_base;
        if s_ufc_rdy='1' or (s_state=FS_SEND and s_beat_cnt(s_beat_cnt'high)='0') then
            --   if s_incr>=NB_16B_LANES then v_data_byte_base := s_data_byte_base + NB_16B_LANES*2;
            --elsif s_incr<NB_16B_LANES  then v_data_byte_base := s_data_byte_base + (s_incr & '0'); end if;
               if s_incr>=NB_16B_LANES then s_data_byte_base <= s_data_byte_base + NB_16B_LANES;
            elsif s_incr<NB_16B_LANES  then s_data_byte_base <= s_data_byte_base + s_incr; end if;
        end if;
        --s_data_byte_base <= v_data_byte_base;
        
        ----------------------------------------------------------------------------------------------------
        -- UFC bus
        ----------------------------------------------------------------------------------------------------
        -- Prepare UFC data bytes
        for i in 0 to 2*NB_16B_LANES-1 loop
            s_ufc_d(i*8+8-1 downto i*8) <= std_logic_vector(s_data_byte_base & '0' + i);
        end loop;
        
        -- Build 'req' output
           if s_state=FS_REQ and s_do_req='1'  then s_ufc_vld <= '1';
        elsif s_state=FS_ACK and s_ufc_rdy='1' then s_ufc_vld <= '0'; end if;
        
        -- Build 'nb' output
        if s_state=FS_REQ and s_do_req='1' then s_ufc_nb  <= s_req_nb; end if;
          
    end if;
    end process p_main;
    
    --##############################################################
    -- Internal / Ports assignments
    --##############################################################
    gen_direct : if ufc_direct = '1' generate
      ufc_vld    <= s_ufc_vld;
      s_ufc_rdy  <= ufc_rdy;
      ufc_nb     <= s_ufc_nb;
      ufc_d      <= s_ufc_d;     
      frame_cnt  <= std_logic_vector(s_frame_cnt);
      ufc_sop    <= '-';              -- not used in Direct Mode
      ufc_eop    <= '-';              -- not used in Direct Mode 
      ufc_bena   <=  (others => '-'); -- not used in Direct Mode
    end generate gen_direct;
    
    gen_streaming : if ufc_direct = '0' generate
      
      -- UFC Ready
      s_ufc_rdy <= '1' when s_tx_fsm = s_idle else '0';
      
      tx_proc : process(clk, rst_n)
        variable v_tx_end  : unsigned(4 downto 0); -- to count up to 16 
        variable v_tx_bena : std_logic_vector(ufc_bena'range); 
      begin
        if rst_n='0' then
          s_fifo_wrreq  <= '0';
          s_fifo_wrdata <= (others => '0');
          s_tx_len      <= (others => '0');
          s_tx_cnt      <= (others => '0');
          s_tx_fsm      <= s_idle;
        elsif rising_edge(clk) then
        
          s_fifo_wrreq  <= '0';
          s_fifo_wrdata <= (others => '0');
          
          case s_tx_fsm is
          
            when s_idle => 
              if s_ufc_vld = '1' and s_ufc_rdy = '1' then
                -- Number of Bytes to send
                s_tx_len <= to_unsigned((to_integer(unsigned(s_ufc_nb))+1)*2, s_tx_len'length); 
                s_tx_cnt <= (others => '0');
                s_tx_fsm <= s_store;
              end if;
              
            when s_store =>
              -- Compute Total Number of Words to send
              v_tx_end := (others => '0');
              case ufc_d'length is
                when 016 => v_tx_end := resize(s_tx_len(4 downto 1), v_tx_end'length);
                when 032 => v_tx_end := resize(s_tx_len(4 downto 2), v_tx_end'length) + s_tx_len(1 downto 1);
                when 064 => v_tx_end := resize(s_tx_len(4 downto 3), v_tx_end'length) + or_ed(s_tx_len(2 downto 1));
                when 128 => v_tx_end := to_unsigned(1, v_tx_end'length);
                when 256 => v_tx_end := to_unsigned(1, v_tx_end'length);
                when others => report "Unsupported mode so far" severity failure;
              end case; 
              -- Compute Byte Enables for last Word
              v_tx_bena := calc_byte_enables(to_integer(s_tx_len), to_integer(v_tx_end), ufc_d'length);
              
              -- Storing Data in Fifo
              s_fifo_wrreq  <= '1';
              s_fifo_wrdata(ufc_d'range) <= s_ufc_d;
              s_tx_cnt <= s_tx_cnt + 1;
              if s_tx_cnt = 0 then
                s_fifo_wrdata(ufc_d'length) <= '1'; -- SOP
              end if;
              if s_tx_cnt = v_tx_end-1 then -- Last Word ? 
                s_fifo_wrdata(ufc_d'length+1) <= '1'; -- EOP
                s_fifo_wrdata(ufc_bena'length+ufc_d'length+1 downto ufc_d'length+2) <= v_tx_bena; -- Byte Enables
                s_tx_fsm <= s_idle;
              end if;
            
          end case;
          
        end if;
      end process;
      
      -- Fifo Reset
      s_fifo_aclr <= not rst_n;
    
      -- Fifo Full check
      process(clk, rst_n)
      begin
        if rst_n = '0' then
        elsif rising_edge(clk) then
          if s_fifo_wrreq = '1' and s_fifo_full = '1' then
            report "[ufc_gen] i_fifo Fifo Full error !" severity failure;
          end if;
        end if;
      end process;                        
  
      -- The Fifo
      i_fifo : scfifo
        generic map (
        	 intended_device_family  => "Arria10",
        	 lpm_type                => "scfifo"       ,
        	 lpm_numwords            => 2**c_fifo_depth,
        	 lpm_width               => c_fifo_wsize   ,
        	 lpm_widthu              => c_fifo_depth   ,
        	 lpm_showahead           => "ON"           , -- ON !
        	 use_eab                 => "OFF"          , -- implemented in Logic !
        	 overflow_checking       => "ON"           ,
        	 underflow_checking      => "ON"           ,
        	 add_ram_output_register => "ON"           
        )
        port map ( 
          aclr  => s_fifo_aclr  , 
          clock => clk          ,
          sclr  => '0'          ,
          wrreq => s_fifo_wrreq , 
          data  => s_fifo_wrdata, 
          full  => s_fifo_full  ,
          usedw => open         ,
          empty => s_fifo_empty , 
          rdreq => s_fifo_rdreq , 
          q     => s_fifo_rddata  
        );
      
      -- Fifo Read Management  
      s_fifo_rdreq <= not s_fifo_empty and ufc_rdy;   
      
      -- Tx Streaming Outputs
      ufc_vld  <= not s_fifo_empty;
      ufc_sop  <= not s_fifo_empty and s_fifo_rddata(ufc_d'length);
      ufc_eop  <= not s_fifo_empty and s_fifo_rddata(ufc_d'length+1);
      ufc_d    <= s_fifo_rddata(ufc_d'range);
      ufc_bena <= s_fifo_rddata(ufc_bena'length+ufc_d'length+1 downto ufc_d'length+2);
      --
      ufc_nb     <= (others => '-'); -- not used in Streaming Mode
      frame_cnt  <= std_logic_vector(s_frame_cnt);
      
    end generate gen_streaming;
    
end architecture rtl;
