----------------------------------------------------------------------------------------------------
-- Copyright (c) ReFLEX CES 1998-2012
--
-- Use of this source code through a simulator and/or a compiler tool
-- is illegal if not authorised through ReFLEX CES License agreement.
----------------------------------------------------------------------------------------------------
-- Project     : 
----------------------------------------------------------------------------------------------------
-- Top level   : top.vhd
-- File        : pdu_gen.vhd
-- Author      : flavenant@reflexces.com
-- Company     : ReFLEX CES
--               2, rue du gevaudan
--               91047 LISSES
--               FRANCE
--               http://www.reflexces.com
-- Plateforme  : Windows XP
-- Simulator   : Mentor Graphics ModelSim
-- Synthesis   : Quartus II
-- Target      : Stratix IV
-- Dependency  :
----------------------------------------------------------------------------------------------------
-- Description :
-- PDU generator
--  
----------------------------------------------------------------------------------------------------
-- Version      Date            Author              Description
-- 0.1          2012/01/16      FLA                 Creation
----------------------------------------------------------------------------------------------------
library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;

entity pdu_gen is
    generic (
      NB_16B_LANES     : integer   :=   1; -- Number of 16bits lanes.
      COUNTER_WIDTH    : integer   :=  32; -- Number of bits for the frame counter
      streaming_mode   : std_logic := '0'  -- Streaming Mode if '1', Framing if '0'
    );                   
    port (
        -- Reset and clocks
          rst_n            : in  std_logic
        ; clk              : in  std_logic
                           
        -- PDU             
        ; pdu_d            : out std_logic_vector(16*NB_16B_LANES-1 downto 0)
        ; pdu_vld          : out std_logic
        ; pdu_bena         : out std_logic_vector(2*NB_16B_LANES-1 downto 0)
        ; pdu_sop          : out std_logic
        ; pdu_eop          : out std_logic
        ; pdu_rdy          : in  std_logic
        
        -- Fixed Mode
        ; fixed_size_on    : in  std_logic                     -- Fixed Frame Mode if '1'                                        
        ; fixed_size_max   : in  std_logic_vector(31 downto 0) -- Fixed Frame Mode Max Size                                          
        ; fixed_size_min   : in  std_logic_vector(31 downto 0) -- Fixed Frame Mode Min Size                                        
        ; fixed_size_incr  : in  std_logic_vector(31 downto 0) -- Fixed Frame Mode Size Increment                                        
        ; fixed_size_ifgap : in  std_logic_vector(31 downto 0) -- Fixed Frame Mode InterFrame Gap                                        
        ; fixed_cut_size   : in  std_logic_vector(31 downto 0) -- Fixed Frame Mode Cut Size                                        
        ; fixed_cut_gap    : in  std_logic_vector(31 downto 0) -- Fixed Frame Mode Cut Gap                                        
      
        -- Control / Status
        ; underflow_mode   : in  std_logic                                  -- '1' to generate not valid cycles
        ; endless_mode     : in  std_logic := '0'                           -- '1' to generate endless frame (no EOP)
        ; enable           : in  std_logic                                  
        ; frame_cnt        : out std_logic_vector(COUNTER_WIDTH-1 downto 0) -- number of frames sent
        ; frame_size       : out std_logic_vector(31 downto 0)              -- current Frame size (when in Fixed Mode)
        ; reset_cnt        : in  std_logic
    );
end entity pdu_gen;

architecture rtl of pdu_gen is
	----------------------------------------------------------------
	-- Type declarations
	----------------------------------------------------------------
    
	----------------------------------------------------------------
	-- Function declarations
	----------------------------------------------------------------
    -- Round to floor Log2
    function MyLocalLog2(nb : integer) return integer is
        variable v_rtn : integer := 0;
    begin
        if nb=0 then v_rtn := 0;
        else         while (nb>2**v_rtn) loop v_rtn := v_rtn + 1; end loop;
        end if;
        return v_rtn;
    end function MyLocalLog2;
        
    -- LFSR (Linear Feedback Shift Register) function for test
    -- /!\ use xnor to set 111...11 as the idle state instead of 000...00
    -- '1' bit in mask enables xnor with corresponding bit in vectIn
    function DoLFSR(vectIn : std_logic_vector;
                    mask   : std_logic_vector ) return std_logic_vector is
        variable v_rtn : std_logic_vector(vectIn'range);
        variable v_fb  : std_logic;
    begin
        v_fb              := vectIn(0);
        v_rtn(v_rtn'high) := v_fb;
    
        for i in vectIn'low to vectIn'high-1 loop
			if (mask(i+1)='1') then v_rtn(i) := vectIn(i+1) xnor v_fb;
			else                    v_rtn(i) := vectIn(i+1);
			end if;
        end loop;
        
        return v_rtn;
    end function DoLFSR;
    
	----------------------------------------------------------------
	-- Component declarations
	----------------------------------------------------------------
    
	----------------------------------------------------------------
	-- Constant declarations
	----------------------------------------------------------------
    constant C_NB_16B_LANES_BITS : integer                       := MyLocalLog2(NB_16B_LANES)+1;
    constant C_LFSR_VLD          : std_logic_vector(28 downto 0) := std_logic_vector(to_unsigned(16#4f3#, 29)); 
    constant C_LFSR_SOP          : std_logic_vector(29 downto 0) := std_logic_vector(to_unsigned(16#3ab#, 30)); 
    constant C_LFSR_EOP          : std_logic_vector(30 downto 0) := std_logic_vector(to_unsigned(16#169#, 31)); 
    constant C_LFSR_CNT          : std_logic_vector(31 downto 0) := std_logic_vector(to_unsigned(16#7c3#, 32)); 
        

	----------------------------------------------------------------
	-- Signal declarations
	----------------------------------------------------------------
    signal s_lfsr_vld  : std_logic_vector(C_LFSR_VLD'range);
    signal s_lfsr_sop  : std_logic_vector(C_LFSR_SOP'range);
    signal s_lfsr_eop  : std_logic_vector(C_LFSR_EOP'range);
    signal s_lfsr_cnt  : std_logic_vector(C_LFSR_CNT'range);
    signal s_sop       : std_logic;
    signal s_eop       : std_logic;
    signal s_vld       : std_logic;
    signal s_in_frame  : std_logic;
    signal s_data_byte : unsigned(7 downto 0);
                       
    signal s_pdu_rdy   : std_logic;
    signal s_pdu_vld   : std_logic;
    signal s_pdu_sop   : std_logic;
    signal s_pdu_d     : unsigned(16*NB_16B_LANES-1 downto 0);
    signal s_pdu_bena  : unsigned(2*NB_16B_LANES-1 downto 0);
    signal s_pdu_eop   : std_logic;
    
    signal s_frame_cnt : unsigned(frame_cnt'range);
    
    signal s_fixed_size_max   : unsigned(fixed_size_max  'range);                                 
    signal s_fixed_size_min   : unsigned(fixed_size_min  'range);                               
    signal s_fixed_size_incr  : unsigned(fixed_size_incr 'range);                                     
    signal s_fixed_size_ifgap : unsigned(fixed_size_ifgap'range);                                     
    signal s_fixed_cut_size   : unsigned(fixed_cut_size  'range);                                    
    signal s_fixed_cut_gap    : unsigned(fixed_cut_gap   'range);                                       
    
    type s_fixed_fsm_t is ( s_idle, s_start, s_data, s_wait_cut, s_wait ); 
    signal s_fixed_fsm    : s_fixed_fsm_t; 
    signal s_fixed_f_size : unsigned(fixed_size_max  'range); 
    signal s_fixed_d_cnt  : unsigned(fixed_size_max  'range); 
    signal s_fixed_w_cnt  : unsigned(fixed_size_ifgap'range); 
    signal s_fixed_c_cnt  : unsigned(fixed_cut_size  'range); 
     
    
begin


    --##############################################################
    -- Main process
    --##############################################################     
    p_main : process (clk, rst_n)
        variable v_lfsr_cnt     : std_logic_vector(s_lfsr_cnt'range);
        variable v_lfsr_vld0    : std_logic;
        variable v_lfsr_sop0    : std_logic;
        variable v_lfsr_eop0    : std_logic;
        variable v_in_frame     : std_logic;
        variable v_sop          : std_logic;
        variable v_eop          : std_logic;
        variable v_vld          : std_logic;
        variable v_idx          : integer range 0 to 2**(2*NB_16B_LANES+1+1);
        variable v_bena         : unsigned(2*NB_16B_LANES+1 downto 0);
        variable v_data_byte    : unsigned(s_data_byte'range);
        variable v_din_d        : unsigned(s_pdu_d'range);
    begin
    if rst_n='0' then 
        s_lfsr_vld     <= C_LFSR_VLD;
        s_lfsr_sop     <= C_LFSR_SOP;
        s_lfsr_eop     <= C_LFSR_EOP;
        s_lfsr_cnt     <= C_LFSR_CNT;
        s_sop          <= '0';
        s_eop          <= '0';
        s_vld          <= '0';
        s_in_frame     <= '0';
        s_data_byte    <= (others=>'0');
        s_pdu_d        <= (others=>'0');
        s_pdu_vld      <= '0';
        s_pdu_bena     <= (others=>'0');
        s_pdu_sop      <= '0';
        s_pdu_eop      <= '0';
        s_frame_cnt    <= (others=>'0');
        --   
        s_fixed_c_cnt  <= (others=>'0');
        s_fixed_d_cnt  <= (others=>'0');
        s_fixed_w_cnt  <= (others=>'0');
        s_fixed_f_size <= (others=>'0');
        s_fixed_fsm    <= s_idle;
        
    elsif rising_edge(clk) then
    
        ----------------------------------------------------------------------------------------------------
        -- Manage frame counter
        ----------------------------------------------------------------------------------------------------
        -- Count frames sent
           if reset_cnt='1' then s_frame_cnt <= (others=>'0');
        elsif streaming_mode='0' then
          if s_pdu_vld='1' and s_pdu_eop='1' and s_pdu_rdy='1' then s_frame_cnt <= s_frame_cnt + 1; end if;
        else -- streaming_mode='0' : makes a pseudo Frame Counter every time the Data make a rollover to 0x0100
          if s_pdu_vld='1' and s_pdu_rdy='1' and s_pdu_d(15 downto 0)=x"0100" then s_frame_cnt <= s_frame_cnt + 1; end if;
        end if;
    
        ----------------------------------------------------------------------------------------------------
        ----------------------------------------------------------------------------------------------------
        -- LFSR MODE
        ----------------------------------------------------------------------------------------------------
        ----------------------------------------------------------------------------------------------------
        if fixed_size_on = '0' then
        
          ----------------------------------------------------------------------------------------------------
          -- Generate LFSR
          ----------------------------------------------------------------------------------------------------
          s_lfsr_vld <= DoLFSR(s_lfsr_vld, C_LFSR_VLD);
          s_lfsr_sop <= DoLFSR(s_lfsr_sop, C_LFSR_SOP);
          s_lfsr_eop <= DoLFSR(s_lfsr_eop, C_LFSR_EOP);
          s_lfsr_cnt <= DoLFSR(s_lfsr_cnt, C_LFSR_CNT);
          
          v_lfsr_vld0 := s_lfsr_vld(0) or s_lfsr_vld(1) or not(underflow_mode);
          v_lfsr_sop0 := (s_lfsr_sop(0) or not(underflow_mode)) and enable;
             if enable='0'       then v_lfsr_eop0 := s_lfsr_eop(0) and s_lfsr_eop(1); -- decrease EOP probability
          elsif endless_mode='1' then v_lfsr_eop0 := '0';
          else                        v_lfsr_eop0 := s_lfsr_eop(0) and s_lfsr_eop(1); end if; 
          v_lfsr_cnt  := s_lfsr_cnt;
          
          ----------------------------------------------------------------------------------------------------
          -- Generate packet stream flags
          ----------------------------------------------------------------------------------------------------
          -- Generate "in frame" state
          v_in_frame := s_in_frame or v_lfsr_sop0;
          
          -- Generate flags
          v_sop  := v_lfsr_sop0 and not(s_in_frame);
          v_eop  := v_lfsr_eop0 and      v_in_frame ;
          v_vld  := (v_lfsr_vld0 and v_in_frame) or v_sop or v_eop;
          
          -- Register flags
          if s_vld='0' or s_pdu_vld='0' or s_pdu_rdy='1' then
              s_in_frame <= v_in_frame and not(v_eop);
              s_sop <= v_sop;
              s_eop <= v_eop;
              s_vld <= v_vld;
          end if;
          
          ----------------------------------------------------------------------------------------------------
          -- Build packet stream
          ----------------------------------------------------------------------------------------------------
          -- Generate an index for byte enable
          v_idx  := to_integer(unsigned(v_lfsr_cnt(C_NB_16B_LANES_BITS downto 0)));
          if v_idx>(2*NB_16B_LANES) then v_idx := 2*NB_16B_LANES; end if;
          if v_idx<1                then v_idx := 1;              end if;
          
          -- Generate byte enable : always all enabled in Streaming Mode
          v_bena        := (others=>'0');
          v_bena(v_idx) := '1';
          v_bena        := v_bena - 1;
          if s_eop='0' or streaming_mode='1' then v_bena := (others=>'1'); end if;
          
          -- Generate data bytes
          v_data_byte := s_data_byte;
          v_din_d     := (others=>'0');
          for i in 0 to 2*NB_16B_LANES-1 loop
              if v_bena(i)='1' then v_din_d(i*8+8-1 downto i*8) := v_data_byte + i; end if;
          end loop;
          if (s_pdu_vld='0' or s_pdu_rdy='1') and s_vld='1' then
              for i in 0 to 2*NB_16B_LANES-1 loop
                  if v_bena(i)='1' then v_data_byte := v_data_byte + 1; end if;
              end loop;
          end if;
          s_data_byte <= v_data_byte;
          
          -- Register values
          if s_pdu_vld='0' or s_pdu_rdy='1' then
              s_pdu_d      <= v_din_d;
              s_pdu_sop    <= s_sop;
              s_pdu_eop    <= s_eop;
              s_pdu_vld    <= s_vld;
            --if s_eop='1' then s_pdu_bena <= v_bena(s_pdu_bena'range); else s_pdu_bena <= (others=>'0'); end if;
              if s_eop='1' then s_pdu_bena <= v_bena(s_pdu_bena'range); else s_pdu_bena <= (others=>'1'); end if;
          end if;
          
        else
        
        ----------------------------------------------------------------------------------------------------
        ----------------------------------------------------------------------------------------------------
        -- INCREMENTING SIZE MODE, as fast as possible (no random cycles on valid)
        ----------------------------------------------------------------------------------------------------
        ----------------------------------------------------------------------------------------------------
          case s_fixed_fsm is
          
            -- Idle : waiting for enable
            when s_idle =>  
              if enable = '1' then
                -- Generate data bytes
                v_data_byte := s_data_byte;
                v_din_d     := (others=>'0');
                for i in 0 to 2*NB_16B_LANES-1 loop
                   v_din_d(i*8+8-1 downto i*8) := v_data_byte + i;
                end loop;
                for i in 0 to 2*NB_16B_LANES-1 loop
                  v_data_byte := v_data_byte + 1;
                end loop;                     
                s_data_byte    <= v_data_byte;
                s_fixed_f_size <= s_fixed_size_max;
                s_fixed_fsm    <= s_start;
              else
                s_fixed_c_cnt  <= (others=>'0');
                s_fixed_d_cnt  <= (others=>'0');
                s_fixed_w_cnt  <= (others=>'0');
                s_fixed_f_size <= (others=>'0');
              end if;
              
             -- Generates the SOP  
            when s_start =>
              s_pdu_sop     <= '1';
              s_pdu_vld     <= '1'; 
              s_pdu_d       <= v_din_d;
              s_pdu_bena    <= (others => '1');    -- Always sending full words in this mode
              s_fixed_c_cnt <= to_unsigned(1, s_fixed_c_cnt'length);
              s_fixed_d_cnt <= s_fixed_f_size - 1; -- Starts with the Frame of Max Size 
              s_fixed_fsm   <= s_data; 
                                
            when s_data =>             
              if s_pdu_rdy = '1' then -- Previous Word accepted
                -- Prepare next word
                v_data_byte := s_data_byte;
                v_din_d     := (others=>'0');
                for i in 0 to 2*NB_16B_LANES-1 loop
                   v_din_d(i*8+8-1 downto i*8) := v_data_byte + i;
                end loop;
                for i in 0 to 2*NB_16B_LANES-1 loop
                  v_data_byte := v_data_byte + 1;
                end loop;
                s_data_byte <= v_data_byte;             
                s_pdu_d     <= v_din_d;
                --
                s_pdu_sop     <= '0'; -- Release SOP
                s_pdu_vld     <= '1'; -- Maintain DAV
                s_fixed_c_cnt <= s_fixed_c_cnt + 1;  -- Counting Data in case we need to cut a PDU Frame
                s_pdu_eop     <= '0'; -- Release EOP (in case)
                if s_fixed_d_cnt > 0 then
                  s_fixed_d_cnt <= s_fixed_d_cnt - 1;
                end if;
                if s_fixed_d_cnt = 1 then  -- 
                  s_pdu_eop <= '1';
                end if;
                if s_pdu_eop = '1' then -- LAST WORD
                  if s_fixed_size_ifgap > 0 then
                    s_fixed_w_cnt <= s_fixed_size_ifgap - 1;
                    if s_fixed_f_size + s_fixed_size_incr > s_fixed_size_max then
                      s_fixed_f_size <= s_fixed_size_min;
                    else
                      s_fixed_f_size <= s_fixed_f_size + s_fixed_size_incr; -- Next Frame Size
                    end if;
                    s_pdu_vld     <= '0';
                    s_fixed_fsm   <= s_wait; 
                  else -- no InterFrame Gap
                    s_fixed_w_cnt <= (others => '0');
                  end if;
                end if;
                -- Have to insert not valid cycles INSIDE a PDU Frame ?
                if s_fixed_d_cnt > 1 and s_fixed_c_cnt = s_fixed_cut_size and s_fixed_cut_size > 0 then
                  s_pdu_vld     <= '0'; -- Disable DAV, since we want to CUT
                  s_fixed_d_cnt <= s_fixed_d_cnt;
                  s_fixed_c_cnt <= s_fixed_cut_gap;
                  s_fixed_fsm   <= s_wait_cut;
                end if;
              end if;
            
            when s_wait_cut =>             
              if s_fixed_c_cnt = 0 then
                s_fixed_c_cnt <= to_unsigned(1, s_fixed_c_cnt'length);
                s_pdu_vld     <= '1'; -- Re-enable DAV
                s_fixed_fsm   <= s_data;
              else
                s_fixed_c_cnt <= s_fixed_c_cnt - 1;
              end if;
            
            when s_wait =>             
              if s_fixed_w_cnt = 0 then  
                if enable = '0' then
                  s_fixed_fsm <= s_idle; 
                else
                  s_fixed_fsm <= s_start; 
                end if;
              else
                s_fixed_w_cnt <= s_fixed_w_cnt - 1;
              end if;
            
          end case;          
                    
        end if;
        
    end if;
    end process p_main;
    
    --##############################################################
    -- Internal / Ports assignments
    --##############################################################  
    pdu_d      <= std_logic_vector(s_pdu_d);
    pdu_vld    <= s_pdu_vld;
    pdu_bena   <= std_logic_vector(s_pdu_bena) when streaming_mode='0' else (others=>'1');
    pdu_sop    <= s_pdu_sop  when streaming_mode='0' else '0'; 
    pdu_eop    <= s_pdu_eop  when streaming_mode='0' else '0'; 
    s_pdu_rdy  <= pdu_rdy; 
    frame_cnt  <= std_logic_vector(s_frame_cnt);
    frame_size <= std_logic_vector(s_fixed_f_size);
    --
    s_fixed_size_max   <= unsigned(fixed_size_max  ); -- Fixed Frame Mode Max Size                                          
    s_fixed_size_min   <= unsigned(fixed_size_min  ); -- Fixed Frame Mode Min Size                                        
    s_fixed_size_incr  <= unsigned(fixed_size_incr ); -- Fixed Frame Mode Size Increment                                        
    s_fixed_size_ifgap <= unsigned(fixed_size_ifgap); -- Fixed Frame Mode InterFrame Gap                                        
    s_fixed_cut_size   <= unsigned(fixed_cut_size  ); -- Fixed Frame Mode Cut Size
    s_fixed_cut_gap    <= unsigned(fixed_cut_gap   ); -- Fixed Frame Mode Cut Gap
    
end architecture rtl;
