-- ----------------------------------------------------------------
--  Copyright    : ALSE - http://alse-fr.com
--  Contact      : info@alse-fr.com
--  Project Name : ALSE - Aurora IP for Altera Devices
--  Block Name   : xil_top_STLV7325
--  Description  : STLV7325 Xilinx Top Level - x1 - 32b - CRC - SFP Master
--  Author       : E.LAURENDEAU
--  Creation     : 10/11/2017
-- ----------------------------------------------------------------

library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;

----------------------------------------------------------------
entity xil_top_stlv7325 is
----------------------------------------------------------------
  generic (
    nb_lanes       : integer   :=  2;  -- Number of XCVR Lanes
    lane_width     : integer   := 16;  -- 16 or 32
    streaming_mode : std_logic := '0'; -- Streaming Mode if '1', Framing if '0'
    crc_enable     : std_logic := '0'  -- CRC Enabled (only Framing Mode)
  );
  port (

    -- External Clock and Reset

    -- External Input Clock - 100Mhz
    ext_clk100mhz   : in  std_logic; -- External input clock @100MHz
    ext_reset       : in  std_logic; -- Button "K2" on board (active low)

    -- XCVR links - STLV7325 SFP
    clk156_p        : in  std_logic;   -- 156.25MHz
    clk156_n        : in  std_logic;   -- 156.25MHz
    sfp_rx_p        : in  std_logic_vector(nb_lanes-1 Downto 0);
    sfp_rx_n        : in  std_logic_vector(nb_lanes-1 Downto 0);
    sfp_tx_p        : out std_logic_vector(nb_lanes-1 Downto 0);
    sfp_tx_n        : out std_logic_vector(nb_lanes-1 Downto 0);

    -- User active high LEDs
    user_led        : out std_logic_vector(7 downto 0)
  );
----------------------------------------------------------------
end entity xil_top_stlv7325;
----------------------------------------------------------------

----------------------------------------------------------------
architecture rtl of xil_top_stlv7325 is
----------------------------------------------------------------

  -- -----------------------
  --  PLL Kintex 7
  -- -----------------------
  component PLL_STLV7325 is
    port (
      locked     : out std_logic;        -- export
      clk_in1    : in  std_logic := 'X'; -- clk
      clk_out1   : out std_logic;        -- clk
      reset      : in  std_logic := 'X'  -- reset
    );
  end component PLL_STLV7325;


  -- ------------------------------------------------------
  --  Differential Input Buffer
  -- ------------------------------------------------------
  component BUFG
    port (
      I : in  STD_LOGIC;
      O : out STD_LOGIC
    );
  end component;

  component vio_src_prb is
    port (
      clk         : in  std_logic;
      probe_in0   : in  std_logic_vector(31 downto 0);
      probe_in1   : in  std_logic_vector(31 downto 0);
      probe_in2   : in  std_logic_vector(31 downto 0);
      probe_in3   : in  std_logic_vector(31 downto 0);
      probe_in4   : in  std_logic_vector(31 downto 0);
      probe_in5   : in  std_logic_vector(31 downto 0);
      probe_in6   : in  std_logic_vector(31 downto 0);
      probe_in7   : in  std_logic_vector(31 downto 0);
      probe_in8   : in  std_logic_vector(31 downto 0);
      probe_in9   : in  std_logic_vector(31 downto 0);
      probe_in10  : in  std_logic_vector(31 downto 0);
      probe_in11  : in  std_logic_vector(31 downto 0);
      probe_in12  : in  std_logic_vector(31 downto 0);
      probe_in13  : in  std_logic_vector(31 downto 0);
      probe_in14  : in  std_logic_vector(31 downto 0);
      probe_in15  : in  std_logic_vector(31 downto 0);
      probe_out0  : out std_logic_vector( 0 downto 0);
      probe_out1  : out std_logic_vector( 0 downto 0);
      probe_out2  : out std_logic_vector( 0 downto 0);
      probe_out3  : out std_logic_vector( 0 downto 0);
      probe_out4  : out std_logic_vector( 0 downto 0);
      probe_out5  : out std_logic_vector(31 downto 0);
      probe_out6  : out std_logic_vector(31 downto 0);
      probe_out7  : out std_logic_vector(31 downto 0);
      probe_out8  : out std_logic_vector(31 downto 0);
      probe_out9  : out std_logic_vector(31 downto 0);
      probe_out10 : out std_logic_vector(31 downto 0);
      probe_out11 : out std_logic_vector(31 downto 0);
      probe_out12 : out std_logic_vector( 0 downto 0)
    );
  end component vio_src_prb;

  -- Vivado Sources & Probes
  signal s_probe_in0   : std_logic_vector(31 downto 0);
  signal s_probe_in1   : std_logic_vector(31 downto 0);
  signal s_probe_in2   : std_logic_vector(31 downto 0);
  signal s_probe_in3   : std_logic_vector(31 downto 0);
  signal s_probe_in4   : std_logic_vector(31 downto 0);
  signal s_probe_in5   : std_logic_vector(31 downto 0);
  signal s_probe_in6   : std_logic_vector(31 downto 0);
  signal s_probe_in7   : std_logic_vector(31 downto 0);
  signal s_probe_in8   : std_logic_vector(31 downto 0);
  signal s_probe_in9   : std_logic_vector(31 downto 0);
  signal s_probe_in10  : std_logic_vector(31 downto 0);
  signal s_probe_in11  : std_logic_vector(31 downto 0);
  signal s_probe_in12  : std_logic_vector(31 downto 0);
  signal s_probe_in13  : std_logic_vector(31 downto 0);
  signal s_probe_in14  : std_logic_vector(31 downto 0);
  signal s_probe_in15  : std_logic_vector(31 downto 0);
  signal s_probe_out0  : std_logic_vector( 0 downto 0);
  signal s_probe_out1  : std_logic_vector( 0 downto 0);
  signal s_probe_out2  : std_logic_vector( 0 downto 0);
  signal s_probe_out3  : std_logic_vector( 0 downto 0);
  signal s_probe_out4  : std_logic_vector( 0 downto 0);
  signal s_probe_out5  : std_logic_vector(31 downto 0);
  signal s_probe_out6  : std_logic_vector(31 downto 0);
  signal s_probe_out7  : std_logic_vector(31 downto 0);
  signal s_probe_out8  : std_logic_vector(31 downto 0);
  signal s_probe_out9  : std_logic_vector(31 downto 0);
  signal s_probe_out10 : std_logic_vector(31 downto 0);
  signal s_probe_out11 : std_logic_vector(31 downto 0);
  signal s_probe_out12 : std_logic_vector( 0 downto 0);

  -- Clock and Reset and Ticks
  signal ext_clk_100_g     : std_logic;
  signal s_ext_reset_s     : std_logic;
  signal s_ext_reset_ss    : std_logic;
  signal s_por_cnt         : unsigned(7 downto 0) := (others => '0');
  signal s_por_rst         : std_logic;
  signal s_por_rst_n       : std_logic;
  signal s_clk100mhz       : std_logic;
  signal s_user_clk        : std_logic;

  signal s_init_clk : std_logic;
  signal s_rst_n    : std_logic;

  signal s_link_up  : std_logic;
  signal s_lane_up  : std_logic_vector(nb_lanes-1 downto 0);

  signal s_pdu_tx_ena       : std_logic;  -- Set to '1' to Enable PDU Sending by Xilinx
  signal s_pdu_underflow    : std_logic;  -- '1' to generate PDU not valid cycles
  signal s_pdu_endless_mode : std_logic;  -- '1' to generate PDU endless frames (no EOP)
  signal s_nfc_tx_ena       : std_logic;  -- Set to '1' to Enable NFC Sending by Xilinx
  signal s_ufc_tx_ena       : std_logic;  -- Set to '1' to Enable UFC Sending by Xilinx
  signal s_reset_cnt        : std_logic;  -- Reset Counters to 0

    -- PDU Fixed Mode
  signal s_pdu_fixed_size_on    : std_logic;                     -- Fixed Frame Mode if '1'
  signal s_pdu_fixed_size_max   : std_logic_vector(31 downto 0); -- Fixed Frame Mode Max Size
  signal s_pdu_fixed_size_min   : std_logic_vector(31 downto 0); -- Fixed Frame Mode Min Size
  signal s_pdu_fixed_size_incr  : std_logic_vector(31 downto 0); -- Fixed Frame Mode Size Increment
  signal s_pdu_fixed_size_ifgap : std_logic_vector(31 downto 0); -- Fixed Frame Mode InterFrame Gap
  signal s_pdu_fixed_cut_size   : std_logic_vector(31 downto 0); -- Fixed Frame Cut Size
  signal s_pdu_fixed_cut_gap    : std_logic_vector(31 downto 0); -- Fixed Frame Cut Gap

  signal s_pdu_tx_frame_size : std_logic_vector(31 downto 0);
  signal s_pdu_tx_frame_cnt  : std_logic_vector(31 downto 0);
  signal s_pdu_rx_frame_cnt  : std_logic_vector(31 downto 0);
  signal s_pdu_rx_error_cnt  : std_logic_vector(31 downto 0);
  signal s_ufc_rx_frame_cnt  : std_logic_vector(31 downto 0);
  signal s_ufc_rx_error_cnt  : std_logic_vector(31 downto 0);
  signal s_nfc_rx_wait_cnt   : std_logic_vector(31 downto 0);

---------
begin
---------

 -- ----------------------------------------------------------
  --  Input Buffer on ext_clk100mhz_p/n
  -- ----------------------------------------------------------
  i_ext_clk100mhz_bufg : BUFG
    port map (
      I  =>   ext_clk100mhz,
      O  =>   ext_clk_100_g
    );

  -- ----------------------------------------------------------
  --  Power On Reset
  -- ----------------------------------------------------------
  process(ext_clk_100_g)
  begin
    if rising_edge(ext_clk_100_g) then
      -- resync the external reset
      s_ext_reset_s  <= not ext_reset;
      s_ext_reset_ss <= s_ext_reset_s;
      if s_ext_reset_ss = '1' then
        s_por_rst <= '1';
        s_por_cnt <= (others => '0');
      elsif s_por_cnt = 2**s_por_cnt'length-2 then
        s_por_rst <= '0';
      else
        s_por_rst <= '1';
        s_por_cnt <= s_por_cnt + 1;
      end if;
    end if;
  end process;
  s_por_rst_n <= not s_por_rst;


  -- ----------------------------------------------------------
  --  VC707 PLL to generate a General 100MHz clock
  -- ----------------------------------------------------------
  i_PLL_STLV7325 : PLL_STLV7325
    port map (
      clk_in1    => ext_clk_100_g ,
      reset      => s_por_rst     ,
      clk_out1 =>  s_clk100mhz, -- 100MHz
      locked     => open
    );



  -- ---------------------------------------
  --  Aurora Tester Instantiation
  -- ---------------------------------------

  -- Clock and reset
  s_rst_n    <= s_por_rst_n;
  s_init_clk <= s_clk100mhz;

  -- The tester
  i_xil_aurora_tester : entity work.xil_aurora_tester_master
    generic map (
      nb_lanes       => nb_lanes      ,
      lane_width     => lane_width    ,
      streaming_mode => streaming_mode,
      crc_enable     => crc_enable
    )
    port map (
      rst_n                => s_rst_n               ,
      init_clk             => s_init_clk            ,
      gt_clk_p             => clk156_p              ,
      gt_clk_n             => clk156_n              ,
      gt_clk_o             => open                  ,
      user_clk             => s_user_clk            ,
      pdu_tx_ena           => s_pdu_tx_ena          ,
      pdu_underflow        => s_pdu_underflow       ,
      pdu_endless_mode     => s_pdu_endless_mode    ,
      nfc_tx_ena           => s_nfc_tx_ena          ,
      ufc_tx_ena           => s_ufc_tx_ena          ,
      reset_cnt            => s_reset_cnt           ,
      pdu_fixed_size_on    => s_pdu_fixed_size_on   ,
      pdu_fixed_size_max   => s_pdu_fixed_size_max  ,
      pdu_fixed_size_min   => s_pdu_fixed_size_min  ,
      pdu_fixed_size_incr  => s_pdu_fixed_size_incr ,
      pdu_fixed_size_ifgap => s_pdu_fixed_size_ifgap,
      pdu_fixed_cut_size   => s_pdu_fixed_cut_size  ,
      pdu_fixed_cut_gap    => s_pdu_fixed_cut_gap   ,
      xcvr_rx_p            => sfp_rx_p              ,
      xcvr_rx_n            => sfp_rx_n              ,
      xcvr_tx_p            => sfp_tx_p              ,
      xcvr_tx_n            => sfp_tx_n              ,
      link_up              => s_link_up             ,
      lane_up              => s_lane_up             ,
      status               => open                  ,
      pdu_tx_frame_size    => s_pdu_tx_frame_size   ,
      pdu_tx_frame_cnt     => s_pdu_tx_frame_cnt    ,
      pdu_rx_frame_cnt     => s_pdu_rx_frame_cnt    ,
      pdu_rx_error_cnt     => s_pdu_rx_error_cnt    ,
      ufc_rx_frame_cnt     => s_ufc_rx_frame_cnt    ,
      ufc_rx_error_cnt     => s_ufc_rx_error_cnt    ,
      nfc_rx_wait_cnt      => s_nfc_rx_wait_cnt
    );


  -- ----------------------------------
  --  Vivado VIO (Sources & Probes)
  -- ----------------------------------
  i_vio_src_prb : vio_src_prb
    port map (
      clk         => s_user_clk   ,
      probe_in0   => s_probe_in0  ,
      probe_in1   => s_probe_in1  ,
      probe_in2   => s_probe_in2  ,
      probe_in3   => s_probe_in3  ,
      probe_in4   => s_probe_in4  ,
      probe_in5   => s_probe_in5  ,
      probe_in6   => s_probe_in6  ,
      probe_in7   => s_probe_in7  ,
      probe_in8   => s_probe_in8  ,
      probe_in9   => s_probe_in9  ,
      probe_in10  => s_probe_in10 ,
      probe_in11  => s_probe_in11 ,
      probe_in12  => s_probe_in12 ,
      probe_in13  => s_probe_in13 ,
      probe_in14  => s_probe_in14 ,
      probe_in15  => s_probe_in15 ,
      probe_out0  => s_probe_out0 ,
      probe_out1  => s_probe_out1 ,
      probe_out2  => s_probe_out2 ,
      probe_out3  => s_probe_out3 ,
      probe_out4  => s_probe_out4 ,
      probe_out5  => s_probe_out5 ,
      probe_out6  => s_probe_out6 ,
      probe_out7  => s_probe_out7 ,
      probe_out8  => s_probe_out8 ,
      probe_out9  => s_probe_out9 ,
      probe_out10 => s_probe_out10,
      probe_out11 => s_probe_out11,
      probe_out12 => s_probe_out12
    );

  -- Source
  s_pdu_tx_ena           <= s_probe_out0(0);
  s_pdu_underflow        <= s_probe_out1(0);
  -- s_nfc_tx_ena           <= s_probe_out2(0);
  s_nfc_tx_ena           <= '0';
  s_ufc_tx_ena           <= s_probe_out3(0);
  s_pdu_fixed_size_on    <= s_probe_out4(0);
  s_pdu_endless_mode     <= s_probe_out5(0);
  s_pdu_fixed_size_max   <= s_probe_out6 ;
  s_pdu_fixed_size_min   <= s_probe_out7 ;
  s_pdu_fixed_size_incr  <= s_probe_out8 ;
  s_pdu_fixed_size_ifgap <= s_probe_out9 ;
  s_pdu_fixed_cut_size   <= s_probe_out10;
  s_pdu_fixed_cut_gap    <= s_probe_out11;
  s_reset_cnt            <= s_probe_out12(0);

  -- Probes
  s_probe_in0  <= s_pdu_rx_frame_cnt ;
  s_probe_in1  <= s_pdu_rx_error_cnt ;
  s_probe_in2  <= s_ufc_rx_frame_cnt ;
  s_probe_in3  <= s_ufc_rx_error_cnt ;
  s_probe_in4  <= s_pdu_tx_frame_size;
  s_probe_in5  <= s_pdu_tx_frame_cnt ;
  s_probe_in6  <= s_probe_out6;
  s_probe_in7  <= s_probe_out7;
  s_probe_in8  <= s_probe_out8;
  s_probe_in9  <= s_probe_out9;
  s_probe_in10 <= s_probe_out10;
  s_probe_in11 <= s_probe_out11;
  s_probe_in12 <= s_nfc_rx_wait_cnt;
  s_probe_in13 <= (others => '0');
  s_probe_in14 <= (others => '0');
  s_probe_in15 <= (others => '0');

	 -- ------------------------------
  --  LEDS Status
	 -- ------------------------------
  user_led(0) <= s_rst_n;
  user_led(1) <= s_link_up;
  user_led(2) <= s_lane_up(0);
  user_led(3) <= '0';
  user_led(4) <= '0';
  user_led(5) <= '0';
  user_led(6) <= '0';
  user_led(7) <= '0';


end architecture rtl;
