///////////////////////////////////////////////////////////////////////////////
// (c) Copyright 2023 Advanced Micro Devices, Inc. All rights reserved.
//
// This file contains confidential and proprietary information
// of Advanced Micro Devices, Inc. and is protected under U.S. and
// international copyright and other intellectual property
// laws.
//
// DISCLAIMER
// This disclaimer is not a license and does not grant any
// rights to the materials distributed herewith. Except as
// otherwise provided in a valid license issued to you by
// AMD, and to the maximum extent permitted by applicable
// law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
// WITH ALL FAULTS, AND AMD HEREBY DISCLAIMS ALL WARRANTIES
// AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
// BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
// INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
// (2) AMD shall not be liable (whether in contract or tort,
// including negligence, or under any other theory of
// liability) for any loss or damage of any kind or nature
// related to, arising under or in connection with these
// materials, including for any direct, or any indirect,
// special, incidental, or consequential loss or damage
// (including loss of data, profits, goodwill, or any type of
// loss or damage suffered as a result of any action brought
// by a third party) even if such damage or loss was
// reasonably foreseeable or AMD had been advised of the
// possibility of the same.
//
// CRITICAL APPLICATIONS
// AMD products are not designed or intended to be fail-
// safe, or for use in any application requiring fail-safe
// performance, such as life-support or safety devices or
// systems, Class III medical devices, nuclear facilities,
// applications related to the deployment of airbags, or any
// other applications that could lead to death, personal
// injury, or severe property or environmental damage
// (individually and collectively, "Critical
// Applications"). Customer assumes the sole risk and
// liability of any use of AMD products in Critical
// Applications, subject only to applicable laws and
// regulations governing limitations on product liability.
//
// THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
// PART OF THIS FILE AT ALL TIMES.
//
//
///////////////////////////////////////////////////////////////////////////////
//
//  RX_LL_UFC_DATAPATH
//
//
//  Description: the RX_LL_UFC_DATAPATH module takes UFC data in Aurora format
//               and transforms it to LocalLink formatted data
//
//               This module supports 1 2-byte lane designs
//
//

`timescale 1 ns / 1 ps

module aurora_8b10b_x1_16b_RX_LL_UFC_DATAPATH
(

    //Traffic Separator Interface
    UFC_DATA,
    UFC_DATA_V,
    UFC_MESSAGE_START,
    UFC_START,


    //LocalLink UFC Interface
    UFC_RX_DATA,
    UFC_RX_REM,
    UFC_RX_SRC_RDY_N,
    UFC_RX_SOF_N,
    UFC_RX_EOF_N,


    //System Interface
    USER_CLK,
    RESET
);

`define DLY #1


//***********************************Port Declarations*******************************


    //Traffic Separator Interface
    input   [0:15]  UFC_DATA;
    input   [0:0]  UFC_DATA_V;
    input   [0:0]  UFC_MESSAGE_START;
    input           UFC_START;


    //LocalLink Interface
    output  [0:15]  UFC_RX_DATA;
    output  [0:0]  UFC_RX_REM;
    output          UFC_RX_SRC_RDY_N;
    output          UFC_RX_SOF_N;
    output          UFC_RX_EOF_N;


    //System Interface
    input                           USER_CLK;
    input                           RESET;


//****************************External Register Declarations**************************

    reg     [0:15]  UFC_RX_DATA;
    reg             UFC_RX_SRC_RDY_N;
    reg             UFC_RX_SOF_N;
    reg             UFC_RX_EOF_N;



//****************************Internal Register Declarations**************************

    reg     [0:15]  ufc_storage_data_r;
    reg             ufc_storage_v_r;
    reg             ufc_start_r;
    reg             ufc_start_delayed_r;



//*********************************Wire Declarations**********************************



//*********************************Main Body of Code**********************************



    //All input goes into a storage register before it is sent on to the output
    always @(posedge USER_CLK)
        ufc_storage_data_r   <=  `DLY    UFC_DATA;


    //Keep track of whether or not there is data in storage
    always @(posedge USER_CLK)
        if(RESET)   ufc_storage_v_r <=  `DLY    1'b0;
        else        ufc_storage_v_r <=  `DLY    UFC_DATA_V;


    //Output data is registered
    always @(posedge USER_CLK)
        UFC_RX_DATA    <=  `DLY    ufc_storage_data_r;


    //Assert the UFC_RX_SRC_RDY_N signal when there is data in storage
    always @(posedge USER_CLK)
        if(RESET)   UFC_RX_SRC_RDY_N   <=  `DLY     1'b1;
        else        UFC_RX_SRC_RDY_N   <=  `DLY     ~ufc_storage_v_r;



    //Hold start of frame until it can be asserted with data
    always @(posedge USER_CLK)
    begin
        ufc_start_r         <=  `DLY    UFC_START;
        ufc_start_delayed_r <=  `DLY    ufc_start_r;
    end

    //Register the start of frame signal for use with the LocalLink output
    always @(posedge USER_CLK)
        UFC_RX_SOF_N    <=  `DLY    ~ufc_start_delayed_r;


    //Assert EOF when the storage goes from full to empty
    always @(posedge USER_CLK)
        UFC_RX_EOF_N    <=  `DLY    !(!UFC_DATA_V & ufc_storage_v_r);


    //REM is always high in the single lane case
    assign  UFC_RX_REM  =   1'b1;

endmodule
